"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getArgValueSuggestions = getArgValueSuggestions;
var _lodash = require("lodash");
var _common = require("@kbn/data-views-plugin/common");
var _public = require("@kbn/data-plugin/public");
var _plugin_services = require("./plugin_services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function getArgValueSuggestions() {
  const indexPatterns = (0, _plugin_services.getIndexPatterns)();
  async function getIndexPattern(functionArgs) {
    const indexPatternArg = functionArgs.find(({
      name
    }) => name === 'index');
    if (!indexPatternArg) {
      // index argument not provided
      return;
    }
    const indexPatternTitle = (0, _lodash.get)(indexPatternArg, 'value.text');
    return (await indexPatterns.find(indexPatternTitle, 1)).find(index => index.title === indexPatternTitle);
  }
  function containsFieldName(partial, field) {
    if (!partial) {
      return true;
    }
    return field.name.includes(partial);
  }

  // Argument value suggestion handlers requiring custom client side code
  // Could not put with function definition since functions are defined on server
  const customHandlers = {
    es: {
      async index(partial) {
        const search = partial ? `${partial}*` : '*';
        const size = 25;
        return (await indexPatterns.find(search, size)).map(({
          title
        }) => ({
          name: title,
          insertText: title
        }));
      },
      async metric(partial, functionArgs) {
        if (!partial || !partial.includes(':')) {
          return [{
            name: 'avg:'
          }, {
            name: 'cardinality:'
          }, {
            name: 'count'
          }, {
            name: 'max:'
          }, {
            name: 'min:'
          }, {
            name: 'percentiles:'
          }, {
            name: 'sum:'
          }];
        }
        const indexPattern = await getIndexPattern(functionArgs);
        if (!indexPattern) {
          return [];
        }
        const valueSplit = partial.split(':');
        return indexPattern.fields.getByType(_public.KBN_FIELD_TYPES.NUMBER).filter(field => field.aggregatable && containsFieldName(valueSplit[1], field) && !(0, _common.isNestedField)(field)).map(field => {
          const suggestionValue = field.name.replaceAll(':', '\\:');
          return {
            name: `${valueSplit[0]}:${suggestionValue}`,
            help: field.type,
            insertText: suggestionValue
          };
        });
      },
      async split(partial, functionArgs) {
        const indexPattern = await getIndexPattern(functionArgs);
        if (!indexPattern) {
          return [];
        }
        return indexPattern.fields.getAll().filter(field => field.aggregatable && [_public.KBN_FIELD_TYPES.NUMBER, _public.KBN_FIELD_TYPES.BOOLEAN, _public.KBN_FIELD_TYPES.DATE, _public.KBN_FIELD_TYPES.IP, _public.KBN_FIELD_TYPES.STRING].includes(field.type) && containsFieldName(partial, field) && !(0, _common.isNestedField)(field)).map(field => ({
          name: field.name,
          help: field.type,
          insertText: field.name
        }));
      },
      async timefield(partial, functionArgs) {
        const indexPattern = await getIndexPattern(functionArgs);
        if (!indexPattern) {
          return [];
        }
        return indexPattern.fields.getByType(_public.KBN_FIELD_TYPES.DATE).filter(field => containsFieldName(partial, field) && !(0, _common.isNestedField)(field)).map(field => ({
          name: field.name,
          insertText: field.name
        }));
      }
    }
  };
  return {
    /**
     * @param {string} functionName - user provided function name containing argument
     * @param {string} argName - user provided argument name
     * @return {boolean} true when dynamic suggestion handler provided for function argument
     */
    hasDynamicSuggestionsForArgument: (functionName, argName) => {
      return customHandlers[functionName] && customHandlers[functionName][argName];
    },
    /**
     * @param {string} functionName - user provided function name containing argument
     * @param {string} argName - user provided argument name
     * @param {object} functionArgs - user provided function arguments parsed ahead of current argument
     * @param {string} partial - user provided argument value
     * @return {array} array of dynamic suggestions matching partial
     */
    getDynamicSuggestionsForArgument: async (functionName, argName, functionArgs, partialInput = '') => {
      // @ts-ignore
      return await customHandlers[functionName][argName](partialInput, functionArgs);
    },
    /**
     * @param {string} partial - user provided argument value
     * @param {array} staticSuggestions - argument value suggestions
     * @return {array} array of static suggestions matching partial
     */
    getStaticSuggestionsForInput: (partialInput = '', staticSuggestions = []) => {
      if (partialInput) {
        return staticSuggestions.filter(suggestion => {
          return suggestion.name.includes(partialInput);
        });
      }
      return staticSuggestions;
    }
  };
}