"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.usePagination = usePagination;
var _react = require("react");
var _alerts_table_context = require("../contexts/alerts_table_context");
var _types = require("../../../../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function usePagination({
  onPageChange,
  pageIndex,
  pageSize
}) {
  const {
    bulkActions: [, updateBulkActionsState]
  } = (0, _react.useContext)(_alerts_table_context.AlertsTableContext);
  const [pagination, setPagination] = (0, _react.useState)({
    pageIndex,
    pageSize
  });
  const [flyoutAlertIndex, setFlyoutAlertIndex] = (0, _react.useState)(-1);
  const onChangePageSize = (0, _react.useCallback)(_pageSize => {
    setPagination(state => ({
      ...state,
      pageSize: _pageSize,
      pageIndex: 0
    }));
    updateBulkActionsState({
      action: _types.BulkActionsVerbs.clear
    });
    onPageChange({
      pageIndex: 0,
      pageSize: _pageSize
    });
  }, [updateBulkActionsState, onPageChange]);
  const onChangePageIndex = (0, _react.useCallback)(_pageIndex => {
    setPagination(state => ({
      ...state,
      pageIndex: _pageIndex
    }));
    updateBulkActionsState({
      action: _types.BulkActionsVerbs.clear
    });
    onPageChange({
      pageIndex: _pageIndex,
      pageSize: pagination.pageSize
    });
  }, [updateBulkActionsState, onPageChange, pagination.pageSize]);
  const onPaginateFlyout = (0, _react.useCallback)(nextPageIndex => {
    setFlyoutAlertIndex(prevFlyoutAlertIndex => {
      if (nextPageIndex < 0) {
        onChangePageIndex(0);
        return 0;
      }
      const actualPageIndex = pagination.pageSize * pagination.pageIndex + prevFlyoutAlertIndex;
      if (nextPageIndex === actualPageIndex) {
        return prevFlyoutAlertIndex;
      }
      const newPageIndex = Math.floor(nextPageIndex / pagination.pageSize);
      const newAlertIndex = nextPageIndex >= pagination.pageSize * newPageIndex ? nextPageIndex - pagination.pageSize * newPageIndex : nextPageIndex;
      onChangePageIndex(newPageIndex);
      return newAlertIndex;
    });
  }, [onChangePageIndex, pagination.pageIndex, pagination.pageSize]);
  (0, _react.useEffect)(() => {
    setPagination(prevPagination => {
      const newPagination = {
        ...prevPagination
      };
      let updated = false;
      if (prevPagination.pageIndex !== pageIndex) {
        updated = true;
        newPagination.pageIndex = pageIndex;
      }
      if (prevPagination.pageSize !== pageSize) {
        updated = true;
        newPagination.pageSize = pageSize;
      }
      if (updated === true) {
        return newPagination;
      }
      return prevPagination;
    });
  }, [pageIndex, pageSize]);
  return {
    pagination,
    onChangePageSize,
    onChangePageIndex,
    onPaginateFlyout,
    flyoutAlertIndex,
    setFlyoutAlertIndex
  };
}