"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TaskRunMetricsAggregator = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _result_type = require("../lib/result_type");
var _task_events = require("../task_events");
var _lib = require("./lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const HDR_HISTOGRAM_MAX = 5400; // 90 minutes
const HDR_HISTOGRAM_BUCKET_SIZE = 10; // 10 seconds
var TaskRunKeys;
(function (TaskRunKeys) {
  TaskRunKeys["SUCCESS"] = "success";
  TaskRunKeys["NOT_TIMED_OUT"] = "not_timed_out";
  TaskRunKeys["TOTAL"] = "total";
})(TaskRunKeys || (TaskRunKeys = {}));
var TaskRunMetricKeys;
(function (TaskRunMetricKeys) {
  TaskRunMetricKeys["OVERALL"] = "overall";
  TaskRunMetricKeys["BY_TYPE"] = "by_type";
})(TaskRunMetricKeys || (TaskRunMetricKeys = {}));
class TaskRunMetricsAggregator {
  constructor() {
    (0, _defineProperty2.default)(this, "counter", new _lib.MetricCounterService(Object.values(TaskRunKeys), TaskRunMetricKeys.OVERALL));
    (0, _defineProperty2.default)(this, "delayHistogram", new _lib.SimpleHistogram(HDR_HISTOGRAM_MAX, HDR_HISTOGRAM_BUCKET_SIZE));
  }
  initialMetric() {
    return (0, _lodash.merge)(this.counter.initialMetrics(), {
      by_type: {},
      overall: {
        delay: {
          counts: [],
          values: []
        }
      }
    });
  }
  collect() {
    return (0, _lodash.merge)(this.counter.collect(), {
      overall: {
        delay: this.delayHistogram.serialize()
      }
    });
  }
  reset() {
    this.counter.reset();
    this.delayHistogram.reset();
  }
  processTaskLifecycleEvent(taskEvent) {
    if ((0, _task_events.isTaskRunEvent)(taskEvent)) {
      this.processTaskRunEvent(taskEvent);
    } else if ((0, _task_events.isTaskManagerStatEvent)(taskEvent)) {
      this.processTaskManagerStatEvent(taskEvent);
    }
  }
  processTaskRunEvent(taskEvent) {
    const {
      task,
      isExpired
    } = (0, _result_type.unwrap)(taskEvent.event);
    const success = (0, _result_type.isOk)(taskEvent.event);
    const taskType = task.taskType.replaceAll('.', '__');
    const taskTypeGroup = (0, _lib.getTaskTypeGroup)(taskType);

    // increment the total counters
    this.incrementCounters(TaskRunKeys.TOTAL, taskType, taskTypeGroup);

    // increment success counters
    if (success) {
      this.incrementCounters(TaskRunKeys.SUCCESS, taskType, taskTypeGroup);
    }

    // increment expired counters
    if (!isExpired) {
      this.incrementCounters(TaskRunKeys.NOT_TIMED_OUT, taskType, taskTypeGroup);
    }
  }
  processTaskManagerStatEvent(taskEvent) {
    if (taskEvent.id === 'runDelay') {
      const delayInSec = Math.round(taskEvent.event.value);
      this.delayHistogram.record(delayInSec);
    }
  }
  incrementCounters(key, taskType, group) {
    this.counter.increment(key, TaskRunMetricKeys.OVERALL);
    this.counter.increment(key, `${TaskRunMetricKeys.BY_TYPE}.${taskType}`);
    if (group) {
      this.counter.increment(key, `${TaskRunMetricKeys.BY_TYPE}.${group}`);
    }
  }
}
exports.TaskRunMetricsAggregator = TaskRunMetricsAggregator;