"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getUsageRecords = exports.getSearchQueryByCloudSecuritySolution = exports.getCloudSecurityUsageRecord = exports.getAssetAggQueryByCloudSecuritySolution = exports.getAssetAggByCloudSecuritySolution = exports.getAggregationByCloudSecuritySolution = void 0;
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getUsageRecords = (assetCountAggregations, cloudSecuritySolution, taskId, tier, projectId, periodSeconds, logger) => {
  const usageRecords = assetCountAggregations.map(assetCountAggregation => {
    const assetCount = assetCountAggregation.unique_assets.value;
    if (assetCount > _constants.AGGREGATION_PRECISION_THRESHOLD) {
      logger.warn(`The number of unique resources for {${cloudSecuritySolution}} is ${assetCount}, which is higher than the AGGREGATION_PRECISION_THRESHOLD of ${_constants.AGGREGATION_PRECISION_THRESHOLD}.`);
    }
    const minTimestamp = new Date(assetCountAggregation.min_timestamp.value_as_string).toISOString();
    const creationTimestamp = new Date().toISOString();
    const subType = cloudSecuritySolution === _constants.CLOUD_DEFEND ? `${_constants.CLOUD_DEFEND}_block_action_enabled_${assetCountAggregation.key_as_string}` : cloudSecuritySolution;
    const usageRecord = {
      id: `${_constants.CLOUD_SECURITY_TASK_TYPE}_${cloudSecuritySolution}_${projectId}_${creationTimestamp}`,
      usage_timestamp: minTimestamp,
      creation_timestamp: creationTimestamp,
      usage: {
        type: _constants.CLOUD_SECURITY_TASK_TYPE,
        sub_type: subType,
        quantity: assetCount,
        period_seconds: periodSeconds
      },
      source: {
        id: taskId,
        instance_group_id: projectId,
        metadata: {
          tier
        }
      }
    };
    return usageRecord;
  });
  return usageRecords;
};
exports.getUsageRecords = getUsageRecords;
const getAggregationByCloudSecuritySolution = cloudSecuritySolution => {
  if (cloudSecuritySolution === _constants.CLOUD_DEFEND) {
    return {
      asset_count_groups: {
        terms: {
          field: 'cloud_defend.block_action_enabled'
        },
        aggs: {
          unique_assets: {
            cardinality: {
              field: _constants.METERING_CONFIGS[cloudSecuritySolution].assets_identifier
            }
          },
          min_timestamp: {
            min: {
              field: '@timestamp'
            }
          }
        }
      }
    };
  }
  return {
    unique_assets: {
      cardinality: {
        field: _constants.METERING_CONFIGS[cloudSecuritySolution].assets_identifier,
        precision_threshold: _constants.AGGREGATION_PRECISION_THRESHOLD
      }
    },
    min_timestamp: {
      min: {
        field: '@timestamp'
      }
    }
  };
};
exports.getAggregationByCloudSecuritySolution = getAggregationByCloudSecuritySolution;
const getSearchQueryByCloudSecuritySolution = (cloudSecuritySolution, searchFrom) => {
  const mustFilters = [];
  if (cloudSecuritySolution === _constants.CLOUD_DEFEND) {
    mustFilters.push({
      range: {
        '@timestamp': {
          gt: searchFrom.toISOString()
        }
      }
    });
  }
  if (cloudSecuritySolution === _constants.CSPM || cloudSecuritySolution === _constants.KSPM || cloudSecuritySolution === _constants.CNVM) {
    mustFilters.push({
      range: {
        '@timestamp': {
          gte: `now-${_constants.ASSETS_SAMPLE_GRANULARITY}`
        }
      }
    });
  }
  if (cloudSecuritySolution === _constants.CSPM || cloudSecuritySolution === _constants.KSPM) {
    mustFilters.push({
      term: {
        'rule.benchmark.posture_type': cloudSecuritySolution
      }
    });
  }
  return {
    bool: {
      must: mustFilters
    }
  };
};
exports.getSearchQueryByCloudSecuritySolution = getSearchQueryByCloudSecuritySolution;
const getAssetAggQueryByCloudSecuritySolution = (cloudSecuritySolution, searchFrom) => {
  const query = getSearchQueryByCloudSecuritySolution(cloudSecuritySolution, searchFrom);
  const aggs = getAggregationByCloudSecuritySolution(cloudSecuritySolution);
  return {
    index: _constants.METERING_CONFIGS[cloudSecuritySolution].index,
    query,
    size: 0,
    aggs
  };
};
exports.getAssetAggQueryByCloudSecuritySolution = getAssetAggQueryByCloudSecuritySolution;
const getAssetAggByCloudSecuritySolution = async (esClient, cloudSecuritySolution, searchFrom) => {
  const assetsAggQuery = getAssetAggQueryByCloudSecuritySolution(cloudSecuritySolution, searchFrom);
  if (cloudSecuritySolution === _constants.CLOUD_DEFEND) {
    const response = await esClient.search(assetsAggQuery);
    if (!response.aggregations || !response.aggregations.asset_count_groups.buckets.length) return [];
    return response.aggregations.asset_count_groups.buckets;
  }
  const response = await esClient.search(assetsAggQuery);
  if (!response.aggregations) return [];
  return [response.aggregations];
};
exports.getAssetAggByCloudSecuritySolution = getAssetAggByCloudSecuritySolution;
const indexHasDataInDateRange = async (esClient, cloudSecuritySolution, searchFrom) => {
  var _response$hits;
  const response = await esClient.search({
    index: _constants.METERING_CONFIGS[cloudSecuritySolution].index,
    size: 1,
    _source: false,
    query: getSearchQueryByCloudSecuritySolution(cloudSecuritySolution, searchFrom)
  }, {
    ignore: [404]
  });
  return ((_response$hits = response.hits) === null || _response$hits === void 0 ? void 0 : _response$hits.hits.length) > 0;
};
const getSearchStartDate = lastSuccessfulReport => {
  const initialDate = new Date();
  const thresholdDate = new Date(initialDate.getTime() - _constants.THRESHOLD_MINUTES * 60 * 1000);
  let lastSuccessfulReport1;
  if (lastSuccessfulReport) {
    lastSuccessfulReport1 = new Date(lastSuccessfulReport);
    const searchFrom = lastSuccessfulReport && lastSuccessfulReport1 > thresholdDate ? lastSuccessfulReport1 : thresholdDate;
    return searchFrom;
  }
  return thresholdDate;
};
const getCloudSecurityUsageRecord = async ({
  esClient,
  projectId,
  taskId,
  lastSuccessfulReport,
  cloudSecuritySolution,
  tier,
  logger
}) => {
  try {
    const searchFrom = getSearchStartDate(lastSuccessfulReport);
    if (!(await indexHasDataInDateRange(esClient, cloudSecuritySolution, searchFrom))) return;
    const periodSeconds = Math.floor((new Date().getTime() - searchFrom.getTime()) / 1000);
    const assetCountAggregations = await getAssetAggByCloudSecuritySolution(esClient, cloudSecuritySolution, searchFrom);
    const usageRecords = await getUsageRecords(assetCountAggregations, cloudSecuritySolution, taskId, tier, projectId, periodSeconds, logger);
    return usageRecords;
  } catch (err) {
    logger.error(`Failed to fetch ${cloudSecuritySolution} metering data ${err}`);
  }
};
exports.getCloudSecurityUsageRecord = getCloudSecurityUsageRecord;