"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.formatNavigationTree = void 0;
var _fp = require("lodash/fp");
var _i18n = require("@kbn/i18n");
var _securitySolutionNavigation = require("@kbn/security-solution-navigation");
var _util = require("../links/util");
var _utils = require("./utils");
var _constants = require("../links/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SECURITY_PROJECT_TITLE = _i18n.i18n.translate('xpack.securitySolutionServerless.nav.solution.title', {
  defaultMessage: 'Security'
});
const formatNavigationTree = (projectNavLinks, bodyCategories, footerCategories) => {
  const [bodyNavItems, footerNavItems] = (0, _fp.partition)(({
    id
  }) => !(0, _util.isBottomNavItemId)(id), projectNavLinks);
  const bodyChildren = addMainLinksPanelOpenerProp(formatNodesFromLinks(bodyNavItems, bodyCategories));
  return {
    body: [{
      type: 'navGroup',
      id: 'security_project_nav',
      title: SECURITY_PROJECT_TITLE,
      icon: 'logoSecurity',
      breadcrumbStatus: 'hidden',
      defaultIsCollapsed: false,
      children: bodyChildren,
      isCollapsible: false
    }],
    footer: formatFooterNodesFromLinks(footerNavItems, footerCategories)
  };
};

// Body
exports.formatNavigationTree = formatNavigationTree;
const formatNodesFromLinks = (projectNavLinks, parentCategories) => {
  const nodes = [];
  if (parentCategories !== null && parentCategories !== void 0 && parentCategories.length) {
    parentCategories.forEach(category => {
      nodes.push(...formatNodesFromLinksWithCategory(projectNavLinks, category));
    }, []);
  } else {
    nodes.push(...formatNodesFromLinksWithoutCategory(projectNavLinks));
  }
  return nodes;
};
const formatNodesFromLinksWithCategory = (projectNavLinks, category) => {
  if (!(category !== null && category !== void 0 && category.linkIds)) {
    return [];
  }
  if (category.linkIds) {
    const children = category.linkIds.reduce((acc, linkId) => {
      const projectNavLink = projectNavLinks.find(({
        id
      }) => id === linkId);
      if (projectNavLink != null) {
        acc.push(createNodeFromProjectNavLink(projectNavLink));
      }
      return acc;
    }, []);
    if (!children.length) {
      return [];
    }
    const id = (0, _securitySolutionNavigation.isTitleLinkCategory)(category) ? getCategoryIdFromLabel(category.label) : undefined;
    return [{
      id,
      ...((0, _securitySolutionNavigation.isTitleLinkCategory)(category) && {
        title: category.label
      }),
      breadcrumbStatus: 'hidden',
      children
    }];
  }
  return [];
};
const formatNodesFromLinksWithoutCategory = projectNavLinks => projectNavLinks.map(projectNavLink => createNodeFromProjectNavLink(projectNavLink));
const createNodeFromProjectNavLink = projectNavLink => {
  const {
    id,
    title,
    links,
    categories,
    disabled
  } = projectNavLink;
  const link = (0, _util.getNavLinkIdFromProjectPageName)(id);
  const node = {
    id,
    link: link,
    title,
    ...((0, _utils.isBreadcrumbHidden)(id) && {
      breadcrumbStatus: 'hidden'
    }),
    ...(disabled && {
      sideNavStatus: 'hidden'
    })
  };
  if (links !== null && links !== void 0 && links.length) {
    node.children = formatNodesFromLinks(links, categories);
  }
  return node;
};

// Footer

const formatFooterNodesFromLinks = (projectNavLinks, parentCategories) => {
  const nodes = [];
  if (parentCategories !== null && parentCategories !== void 0 && parentCategories.length) {
    parentCategories.forEach(category => {
      if ((0, _securitySolutionNavigation.isSeparatorLinkCategory)(category)) {
        nodes.push(...category.linkIds.reduce((acc, linkId) => {
          const projectNavLink = projectNavLinks.find(({
            id
          }) => id === linkId);
          if (projectNavLink != null) {
            acc.push({
              type: 'navItem',
              link: (0, _util.getNavLinkIdFromProjectPageName)(projectNavLink.id),
              title: projectNavLink.title,
              icon: projectNavLink.sideNavIcon
            });
          }
          return acc;
        }, []));
      }
      if ((0, _securitySolutionNavigation.isAccordionLinkCategory)(category)) {
        var _category$linkIds$red, _category$linkIds;
        nodes.push({
          type: 'navGroup',
          id: getCategoryIdFromLabel(category.label),
          title: category.label,
          icon: category.iconType,
          breadcrumbStatus: 'hidden',
          children: (_category$linkIds$red = (_category$linkIds = category.linkIds) === null || _category$linkIds === void 0 ? void 0 : _category$linkIds.reduce((acc, linkId) => {
            const projectNavLink = projectNavLinks.find(({
              id
            }) => id === linkId);
            if (projectNavLink != null) {
              acc.push({
                title: projectNavLink.title,
                ...((0, _util.isCloudLink)(projectNavLink.id) ? {
                  cloudLink: getCloudLink(projectNavLink.id),
                  openInNewTab: true
                } : {
                  link: (0, _util.getNavLinkIdFromProjectPageName)(projectNavLink.id)
                })
              });
            }
            return acc;
          }, [])) !== null && _category$linkIds$red !== void 0 ? _category$linkIds$red : []
        });
      }
    }, []);
  }
  return nodes;
};

// Utils

const getCategoryIdFromLabel = label => `category-${label.toLowerCase().replace(' ', '_')}`;

/**
 * Adds the `renderAs: 'panelOpener'` prop to the main links that have children
 * This function expects all main links to be in nested groups to add the separation between them.
 * If these "separator" groups change this function will need to be updated.
 */
const addMainLinksPanelOpenerProp = nodes => nodes.map(node => {
  var _node$children;
  if ((_node$children = node.children) !== null && _node$children !== void 0 && _node$children.length) {
    return {
      ...node,
      children: node.children.map(child => ({
        ...child,
        ...(child.children && {
          renderAs: 'panelOpener'
        })
      }))
    };
  }
  return node;
});

/** Returns the cloud link entry the default navigation expects */
const getCloudLink = id => {
  switch (id) {
    case _constants.ExternalPageName.cloudUsersAndRoles:
      return 'userAndRoles';
    case _constants.ExternalPageName.cloudBilling:
      return 'billingAndSub';
    default:
      return undefined;
  }
};