"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateRiskScores = void 0;
var _lodash = require("lodash");
var _technical_rule_data_field_names = require("@kbn/rule-registry-plugin/common/technical_rule_data_field_names");
var _risk_engine = require("../../../../common/risk_engine");
var _with_security_span = require("../../../utils/with_security_span");
var _helpers = require("./helpers");
var _risk_weights = require("./risk_weights");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bucketToResponse = ({
  bucket,
  now,
  identifierField
}) => ({
  '@timestamp': now,
  id_field: identifierField,
  id_value: bucket.key[identifierField],
  calculated_level: bucket.risk_details.value.level,
  calculated_score: bucket.risk_details.value.score,
  calculated_score_norm: bucket.risk_details.value.normalized_score,
  category_1_score: bucket.risk_details.value.category_1_score,
  category_1_count: bucket.risk_details.value.category_1_count,
  notes: bucket.risk_details.value.notes,
  inputs: bucket.inputs.hits.hits.map(riskInput => {
    var _riskInput$fields$ALE, _riskInput$fields, _riskInput$fields$ALE2, _riskInput$fields$ALE3, _riskInput$fields2, _riskInput$fields2$AL, _riskInput$fields$Ti, _riskInput$fields3, _riskInput$fields3$T;
    return {
      id: riskInput._id,
      index: riskInput._index,
      description: `Alert from Rule: ${(_riskInput$fields$ALE = (_riskInput$fields = riskInput.fields) === null || _riskInput$fields === void 0 ? void 0 : (_riskInput$fields$ALE2 = _riskInput$fields[_technical_rule_data_field_names.ALERT_RULE_NAME]) === null || _riskInput$fields$ALE2 === void 0 ? void 0 : _riskInput$fields$ALE2[0]) !== null && _riskInput$fields$ALE !== void 0 ? _riskInput$fields$ALE : 'RULE_NOT_FOUND'}`,
      category: _risk_engine.RiskCategories.category_1,
      risk_score: (_riskInput$fields$ALE3 = (_riskInput$fields2 = riskInput.fields) === null || _riskInput$fields2 === void 0 ? void 0 : (_riskInput$fields2$AL = _riskInput$fields2[_technical_rule_data_field_names.ALERT_RISK_SCORE]) === null || _riskInput$fields2$AL === void 0 ? void 0 : _riskInput$fields2$AL[0]) !== null && _riskInput$fields$ALE3 !== void 0 ? _riskInput$fields$ALE3 : undefined,
      timestamp: (_riskInput$fields$Ti = (_riskInput$fields3 = riskInput.fields) === null || _riskInput$fields3 === void 0 ? void 0 : (_riskInput$fields3$T = _riskInput$fields3['@timestamp']) === null || _riskInput$fields3$T === void 0 ? void 0 : _riskInput$fields3$T[0]) !== null && _riskInput$fields$Ti !== void 0 ? _riskInput$fields$Ti : undefined
    };
  })
});
const filterFromRange = range => ({
  range: {
    '@timestamp': {
      lt: range.end,
      gte: range.start
    }
  }
});
const buildReduceScript = ({
  globalIdentifierTypeWeight
}) => {
  return `
    Map results = new HashMap();
    List inputs = [];
    for (state in states) {
      inputs.addAll(state.inputs)
    }
    Collections.sort(inputs, (a, b) -> b.get('weighted_score').compareTo(a.get('weighted_score')));

    double num_inputs_to_score = Math.min(inputs.length, params.max_risk_inputs_per_identity);
    results['notes'] = [];
    if (num_inputs_to_score == params.max_risk_inputs_per_identity) {
      results['notes'].add('Number of risk inputs (' + inputs.length + ') exceeded the maximum allowed (' + params.max_risk_inputs_per_identity + ').');
    }

    ${(0, _risk_weights.buildCategoryScoreDeclarations)()}
    ${(0, _risk_weights.buildCategoryCountDeclarations)()}

    double total_score = 0;
    double current_score = 0;
    for (int i = 0; i < num_inputs_to_score; i++) {
      current_score = inputs[i].weighted_score / Math.pow(i + 1, params.p);

      ${(0, _risk_weights.buildCategoryAssignment)()}
      total_score += current_score;
    }

    ${globalIdentifierTypeWeight != null ? `total_score *= ${globalIdentifierTypeWeight};` : ''}
    double score_norm = 100 * total_score / params.risk_cap;
    results['score'] = total_score;
    results['normalized_score'] = score_norm;

    if (score_norm < 20) {
      results['level'] = 'Unknown'
    }
    else if (score_norm >= 20 && score_norm < 40) {
      results['level'] = 'Low'
    }
    else if (score_norm >= 40 && score_norm < 70) {
      results['level'] = 'Moderate'
    }
    else if (score_norm >= 70 && score_norm < 90) {
      results['level'] = 'High'
    }
    else if (score_norm >= 90) {
      results['level'] = 'Critical'
    }

    return results;
  `;
};
const buildIdentifierTypeAggregation = ({
  afterKeys,
  identifierType,
  pageSize,
  weights
}) => {
  const globalIdentifierTypeWeight = (0, _risk_weights.getGlobalWeightForIdentifierType)({
    identifierType,
    weights
  });
  const identifierField = (0, _helpers.getFieldForIdentifierAgg)(identifierType);
  return {
    composite: {
      size: pageSize,
      sources: [{
        [identifierField]: {
          terms: {
            field: identifierField
          }
        }
      }],
      after: (0, _helpers.getAfterKeyForIdentifierType)({
        identifierType,
        afterKeys
      })
    },
    aggs: {
      inputs: {
        top_hits: {
          size: 5,
          sort: {
            [_technical_rule_data_field_names.ALERT_RISK_SCORE]: 'desc'
          },
          _source: false,
          docvalue_fields: ['@timestamp', _technical_rule_data_field_names.ALERT_RISK_SCORE, _technical_rule_data_field_names.ALERT_RULE_NAME]
        }
      },
      risk_details: {
        scripted_metric: {
          init_script: 'state.inputs = []',
          map_script: `
              Map fields = new HashMap();
              String category = doc['${_technical_rule_data_field_names.EVENT_KIND}'].value;
              double score = doc['${_technical_rule_data_field_names.ALERT_RISK_SCORE}'].value;
              double weighted_score = 0.0;

              fields.put('time', doc['@timestamp'].value);
              fields.put('category', category);
              fields.put('score', score);
              ${(0, _risk_weights.buildWeightingOfScoreByCategory)({
            userWeights: weights,
            identifierType
          })}
              fields.put('weighted_score', weighted_score);

              state.inputs.add(fields);
            `,
          combine_script: 'return state;',
          params: {
            max_risk_inputs_per_identity: 999999,
            p: 1.5,
            risk_cap: 261.2
          },
          reduce_script: buildReduceScript({
            globalIdentifierTypeWeight
          })
        }
      }
    }
  };
};
const calculateRiskScores = async ({
  afterKeys: userAfterKeys,
  debug,
  esClient,
  filter: userFilter,
  identifierType,
  index,
  logger,
  pageSize,
  range,
  runtimeMappings,
  weights
}) => (0, _with_security_span.withSecuritySpan)('calculateRiskScores', async () => {
  var _response$aggregation, _response$aggregation2, _response$aggregation3, _response$aggregation4, _response$aggregation5, _response$aggregation6;
  const now = new Date().toISOString();
  const filter = [filterFromRange(range), {
    bool: {
      must_not: {
        term: {
          [_technical_rule_data_field_names.ALERT_WORKFLOW_STATUS]: 'closed'
        }
      }
    }
  }, {
    exists: {
      field: _technical_rule_data_field_names.ALERT_RISK_SCORE
    }
  }];
  if (!(0, _lodash.isEmpty)(userFilter)) {
    filter.push(userFilter);
  }
  const identifierTypes = identifierType ? [identifierType] : ['host', 'user'];
  const request = {
    size: 0,
    _source: false,
    index,
    runtime_mappings: runtimeMappings,
    query: {
      bool: {
        filter
      }
    },
    aggs: identifierTypes.reduce((aggs, _identifierType) => {
      aggs[_identifierType] = buildIdentifierTypeAggregation({
        afterKeys: userAfterKeys,
        identifierType: _identifierType,
        pageSize,
        weights
      });
      return aggs;
    }, {})
  };
  if (debug) {
    logger.info(`Executing Risk Score query:\n${JSON.stringify(request)}`);
  }
  const response = await esClient.search(request);
  if (debug) {
    logger.info(`Received Risk Score response:\n${JSON.stringify(response)}`);
  }
  if (response.aggregations == null) {
    return {
      ...(debug ? {
        request,
        response
      } : {}),
      after_keys: {},
      scores: {
        host: [],
        user: []
      }
    };
  }
  const userBuckets = (_response$aggregation = (_response$aggregation2 = response.aggregations.user) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.buckets) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];
  const hostBuckets = (_response$aggregation3 = (_response$aggregation4 = response.aggregations.host) === null || _response$aggregation4 === void 0 ? void 0 : _response$aggregation4.buckets) !== null && _response$aggregation3 !== void 0 ? _response$aggregation3 : [];
  const afterKeys = {
    host: (_response$aggregation5 = response.aggregations.host) === null || _response$aggregation5 === void 0 ? void 0 : _response$aggregation5.after_key,
    user: (_response$aggregation6 = response.aggregations.user) === null || _response$aggregation6 === void 0 ? void 0 : _response$aggregation6.after_key
  };
  return {
    ...(debug ? {
      request,
      response
    } : {}),
    after_keys: afterKeys,
    scores: {
      host: hostBuckets.map(bucket => bucketToResponse({
        bucket,
        identifierField: 'host.name',
        now
      })),
      user: userBuckets.map(bucket => bucketToResponse({
        bucket,
        identifierField: 'user.name',
        now
      }))
    }
  };
});
exports.calculateRiskScores = calculateRiskScores;