"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findThresholdSignals = void 0;
var _lodash = require("lodash");
var _single_search_after = require("../utils/single_search_after");
var _build_threshold_aggregation = require("./build_threshold_aggregation");
var _utils = require("./utils");
var _utils2 = require("../utils/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const hasThresholdFields = threshold => !!threshold.field.length;
const findThresholdSignals = async ({
  from,
  to,
  maxSignals,
  inputIndexPattern,
  services,
  ruleExecutionLogger,
  filter,
  threshold,
  runtimeMappings,
  primaryTimestamp,
  secondaryTimestamp,
  aggregatableTimestampField
}) => {
  // Leaf aggregations used below
  const buckets = [];
  const searchAfterResults = {
    searchDurations: [],
    searchErrors: []
  };
  const warnings = [];
  const includeCardinalityFilter = (0, _utils.shouldFilterByCardinality)(threshold);
  if (hasThresholdFields(threshold)) {
    let sortKeys;
    do {
      const {
        searchResult,
        searchDuration,
        searchErrors
      } = await (0, _single_search_after.singleSearchAfter)({
        aggregations: (0, _build_threshold_aggregation.buildThresholdMultiBucketAggregation)({
          threshold,
          aggregatableTimestampField,
          sortKeys
        }),
        index: inputIndexPattern,
        searchAfterSortIds: undefined,
        from,
        to,
        services,
        ruleExecutionLogger,
        filter,
        pageSize: 0,
        sortOrder: 'desc',
        runtimeMappings,
        primaryTimestamp,
        secondaryTimestamp
      });
      searchAfterResults.searchDurations.push(searchDuration);
      if (!(0, _lodash.isEmpty)(searchErrors)) {
        searchAfterResults.searchErrors.push(...searchErrors);
        sortKeys = undefined; // this will eject us out of the loop
        // if a search failure occurs on a secondary iteration,
        // we will return early.
      } else if ((0, _utils.searchResultHasAggs)(searchResult)) {
        var _searchResult$aggrega, _ref, _searchResult$aggrega2;
        const thresholdTerms = (_searchResult$aggrega = searchResult.aggregations) === null || _searchResult$aggrega === void 0 ? void 0 : _searchResult$aggrega.thresholdTerms;
        sortKeys = thresholdTerms === null || thresholdTerms === void 0 ? void 0 : thresholdTerms.after_key;
        buckets.push(...((_ref = (_searchResult$aggrega2 = searchResult.aggregations) === null || _searchResult$aggrega2 === void 0 ? void 0 : _searchResult$aggrega2.thresholdTerms.buckets) !== null && _ref !== void 0 ? _ref : []));
      } else {
        throw new Error('Aggregations were missing on threshold rule search result');
      }
    } while (sortKeys && buckets.length <= maxSignals);
  } else {
    const {
      searchResult,
      searchDuration,
      searchErrors
    } = await (0, _single_search_after.singleSearchAfter)({
      aggregations: (0, _build_threshold_aggregation.buildThresholdSingleBucketAggregation)({
        threshold,
        aggregatableTimestampField
      }),
      searchAfterSortIds: undefined,
      index: inputIndexPattern,
      from,
      to,
      services,
      ruleExecutionLogger,
      filter,
      pageSize: 0,
      sortOrder: 'desc',
      trackTotalHits: true,
      runtimeMappings,
      primaryTimestamp,
      secondaryTimestamp
    });
    searchAfterResults.searchDurations.push(searchDuration);
    searchAfterResults.searchErrors.push(...searchErrors);
    if (!(0, _utils.searchResultHasAggs)(searchResult) && (0, _lodash.isEmpty)(searchErrors)) {
      throw new Error('Aggregations were missing on threshold rule search result');
    } else if ((0, _utils.searchResultHasAggs)(searchResult)) {
      var _searchResult$aggrega3, _searchResult$aggrega4, _searchResult$aggrega5;
      const docCount = searchResult.hits.total.value;
      if (docCount >= threshold.value && (!includeCardinalityFilter || ((_searchResult$aggrega3 = searchResult === null || searchResult === void 0 ? void 0 : (_searchResult$aggrega4 = searchResult.aggregations) === null || _searchResult$aggrega4 === void 0 ? void 0 : (_searchResult$aggrega5 = _searchResult$aggrega4.cardinality_count) === null || _searchResult$aggrega5 === void 0 ? void 0 : _searchResult$aggrega5.value) !== null && _searchResult$aggrega3 !== void 0 ? _searchResult$aggrega3 : 0) >= threshold.cardinality[0].value)) {
        var _searchResult$aggrega6, _searchResult$aggrega7, _searchResult$aggrega8, _searchResult$aggrega9, _searchResult$aggrega10;
        buckets.push({
          doc_count: docCount,
          key: {},
          max_timestamp: (_searchResult$aggrega6 = (_searchResult$aggrega7 = searchResult.aggregations) === null || _searchResult$aggrega7 === void 0 ? void 0 : _searchResult$aggrega7.max_timestamp) !== null && _searchResult$aggrega6 !== void 0 ? _searchResult$aggrega6 : {
            value: null
          },
          min_timestamp: (_searchResult$aggrega8 = (_searchResult$aggrega9 = searchResult.aggregations) === null || _searchResult$aggrega9 === void 0 ? void 0 : _searchResult$aggrega9.min_timestamp) !== null && _searchResult$aggrega8 !== void 0 ? _searchResult$aggrega8 : {
            value: null
          },
          ...(includeCardinalityFilter ? {
            cardinality_count: (_searchResult$aggrega10 = searchResult.aggregations) === null || _searchResult$aggrega10 === void 0 ? void 0 : _searchResult$aggrega10.cardinality_count
          } : {})
        });
      }
    }
  }
  if (buckets.length > maxSignals) {
    warnings.push((0, _utils2.getMaxSignalsWarning)());
  }
  return {
    buckets: buckets.slice(0, maxSignals),
    ...searchAfterResults,
    warnings
  };
};
exports.findThresholdSignals = findThresholdSignals;