"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.esqlExecutor = void 0;
var _perf_hooks = require("perf_hooks");
var _securitysolutionUtils = require("@kbn/securitysolution-utils");
var _build_esql_search_request = require("./build_esql_search_request");
var _esql_request = require("./esql_request");
var _wrap_esql_alerts = require("./wrap_esql_alerts");
var _enrichments = require("../utils/enrichments");
var _utils = require("./utils");
var _fetch_source_documents = require("./fetch_source_documents");
var _utils2 = require("../utils/utils");
var _with_security_span = require("../../../../utils/with_security_span");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * ES|QL returns results as a single page. max size of 10,000
 * while we try increase size of the request to catch all events
 * we don't want to overload ES/Kibana with large responses
 */
const ESQL_PAGE_SIZE_CIRCUIT_BREAKER = 1000;
const esqlExecutor = async ({
  runOpts: {
    completeRule,
    tuple,
    ruleExecutionLogger,
    bulkCreate,
    mergeStrategy,
    primaryTimestamp,
    secondaryTimestamp,
    exceptionFilter,
    unprocessedExceptions,
    alertTimestampOverride,
    publicBaseUrl
  },
  services,
  state,
  spaceId
}) => {
  const ruleParams = completeRule.ruleParams;
  return (0, _with_security_span.withSecuritySpan)('esqlExecutor', async () => {
    const result = (0, _utils2.createSearchAfterReturnType)();
    let size = tuple.maxSignals;
    while (result.createdSignalsCount <= tuple.maxSignals && size <= ESQL_PAGE_SIZE_CIRCUIT_BREAKER) {
      const esqlRequest = (0, _build_esql_search_request.buildEsqlSearchRequest)({
        query: ruleParams.query,
        from: tuple.from.toISOString(),
        to: tuple.to.toISOString(),
        size,
        filters: [],
        primaryTimestamp,
        secondaryTimestamp,
        exceptionFilter
      });
      ruleExecutionLogger.debug(`ES|QL query request: ${JSON.stringify(esqlRequest)}`);
      const exceptionsWarning = (0, _utils2.getUnprocessedExceptionsWarnings)(unprocessedExceptions);
      if (exceptionsWarning) {
        result.warningMessages.push(exceptionsWarning);
      }
      const esqlSignalSearchStart = _perf_hooks.performance.now();
      const response = await (0, _esql_request.performEsqlRequest)({
        esClient: services.scopedClusterClient.asCurrentUser,
        requestParams: esqlRequest
      });
      const esqlSearchDuration = (0, _utils2.makeFloatString)(_perf_hooks.performance.now() - esqlSignalSearchStart);
      result.searchAfterTimes.push(esqlSearchDuration);
      ruleExecutionLogger.debug(`ES|QL query request took: ${esqlSearchDuration}ms`);
      const isRuleAggregating = (0, _securitysolutionUtils.computeIsESQLQueryAggregating)(completeRule.ruleParams.query);
      const results = response.values
      // slicing already processed results in previous iterations
      .slice(size - tuple.maxSignals).map(row => (0, _utils.rowToDocument)(response.columns, row));
      const index = (0, _securitysolutionUtils.getIndexListFromEsqlQuery)(completeRule.ruleParams.query);
      const sourceDocuments = await (0, _fetch_source_documents.fetchSourceDocuments)({
        esClient: services.scopedClusterClient.asCurrentUser,
        results,
        index,
        isRuleAggregating
      });
      const wrappedAlerts = (0, _wrap_esql_alerts.wrapEsqlAlerts)({
        sourceDocuments,
        isRuleAggregating,
        results,
        spaceId,
        completeRule,
        mergeStrategy,
        alertTimestampOverride,
        ruleExecutionLogger,
        publicBaseUrl,
        tuple
      });
      const enrichAlerts = (0, _enrichments.createEnrichEventsFunction)({
        services,
        logger: ruleExecutionLogger
      });
      const bulkCreateResult = await bulkCreate(wrappedAlerts, tuple.maxSignals - result.createdSignalsCount, enrichAlerts);
      (0, _utils2.addToSearchAfterReturn)({
        current: result,
        next: bulkCreateResult
      });
      ruleExecutionLogger.debug(`Created ${bulkCreateResult.createdItemsCount} alerts`);
      if (bulkCreateResult.alertsWereTruncated) {
        result.warningMessages.push((0, _utils2.getMaxSignalsWarning)());
        break;
      }

      // no more results will be found
      if (response.values.length < size) {
        ruleExecutionLogger.debug(`End of search: Found ${response.values.length} results with page size ${size}`);
        break;
      }
      // ES|QL does not support pagination so we need to increase size of response to be able to catch all events
      size += tuple.maxSignals;
    }
    return {
      ...result,
      state
    };
  });
};
exports.esqlExecutor = esqlExecutor;