"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getExportByObjectIds = void 0;
var _pMap = _interopRequireDefault(require("p-map"));
var _lodash = require("lodash");
var _securitysolutionUtils = require("@kbn/securitysolution-utils");
var _with_security_span = require("../../../../../utils/with_security_span");
var _rule_converters = require("../../normalization/rule_converters");
var _rule_schema = require("../../../rule_schema");
var _find_rules = require("../search/find_rules");
var _get_export_rule_exceptions = require("./get_export_rule_exceptions");
var _get_export_details_ndjson = require("./get_export_details_ndjson");
var _get_export_rule_action_connectors = require("./get_export_rule_action_connectors");
var _transform_rule_to_exportable_format = require("./transform_rule_to_exportable_format");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getExportByObjectIds = async (rulesClient, exceptionsClient, ruleIds, actionsExporter, request, actionsClient) => (0, _with_security_span.withSecuritySpan)('getExportByObjectIds', async () => {
  const rulesAndErrors = await fetchRulesByIds(rulesClient, ruleIds);
  const {
    rules,
    missingRuleIds
  } = rulesAndErrors;

  // Retrieve exceptions
  const exceptions = rules.flatMap(rule => {
    var _rule$exceptions_list;
    return (_rule$exceptions_list = rule.exceptions_list) !== null && _rule$exceptions_list !== void 0 ? _rule$exceptions_list : [];
  });
  const {
    exportData: exceptionLists,
    exportDetails: exceptionDetails
  } = await (0, _get_export_rule_exceptions.getRuleExceptionsForExport)(exceptions, exceptionsClient);

  // Retrieve Action-Connectors
  const {
    actionConnectors,
    actionConnectorDetails
  } = await (0, _get_export_rule_action_connectors.getRuleActionConnectorsForExport)(rules, actionsExporter, request, actionsClient);
  const rulesNdjson = (0, _securitysolutionUtils.transformDataToNdjson)(rules);
  const exportDetails = (0, _get_export_details_ndjson.getExportDetailsNdjson)(rules, missingRuleIds, exceptionDetails, actionConnectorDetails);
  return {
    rulesNdjson,
    exportDetails,
    exceptionLists,
    actionConnectors
  };
});
exports.getExportByObjectIds = getExportByObjectIds;
const fetchRulesByIds = async (rulesClient, ruleIds) => {
  // It's important to avoid too many clauses in the request otherwise ES will fail to process the request
  // with `too_many_clauses` error (see https://github.com/elastic/kibana/issues/170015). The clauses limit
  // used to be set via `indices.query.bool.max_clause_count` but it's an option anymore. The limit is [calculated
  // dynamically](https://www.elastic.co/guide/en/elasticsearch/reference/current/search-settings.html) based
  // on available CPU and memory but the minimum value is 1024.
  // 1024 chunk size helps to solve the problem and use the maximum safe number of clauses.
  const CHUNK_SIZE = 1024;
  // We need to decouple from the number of existing rules and let the complexity/load depend only on the number of users
  // exporting rules simultaneously. By using limited parallelization via p-map we trade speed for stability and scalability.
  const CHUNKS_PROCESSED_IN_PARALLEL = 2;
  const processChunk = async ids => (0, _with_security_span.withSecuritySpan)('processChunk', async () => {
    const rulesResult = await (0, _find_rules.findRules)({
      rulesClient,
      filter: `alert.attributes.params.ruleId: (${ids.join(' OR ')})`,
      page: 1,
      fields: undefined,
      perPage: CHUNK_SIZE,
      sortField: undefined,
      sortOrder: undefined
    });
    const rulesMap = new Map();
    for (const rule of rulesResult.data) {
      rulesMap.set(rule.params.ruleId, rule);
    }
    const rulesAndErrors = ids.map(ruleId => {
      const matchingRule = rulesMap.get(ruleId);
      return matchingRule != null && (0, _rule_schema.hasValidRuleType)(matchingRule) && matchingRule.params.immutable !== true ? {
        rule: (0, _transform_rule_to_exportable_format.transformRuleToExportableFormat)((0, _rule_converters.internalRuleToAPIResponse)(matchingRule))
      } : {
        missingRuleId: ruleId
      };
    });
    return rulesAndErrors;
  });
  const ruleIdChunks = (0, _lodash.chunk)(ruleIds, CHUNK_SIZE);
  // We expect all rules to be processed here to avoid any situation when export of some rules failed silently.
  // If some error happens it just bubbles up as is and processed in the upstream code.
  const rulesAndErrorsChunks = await (0, _pMap.default)(ruleIdChunks, processChunk, {
    concurrency: CHUNKS_PROCESSED_IN_PARALLEL
  });
  const missingRuleIds = [];
  const rules = [];
  for (const rulesAndErrors of rulesAndErrorsChunks) {
    for (const response of rulesAndErrors) {
      if (response.missingRuleId) {
        missingRuleIds.push(response.missingRuleId);
      }
      if (response.rule) {
        rules.push(response.rule);
      }
    }
  }
  return {
    rules,
    missingRuleIds
  };
};