"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ruleParamsModifier = exports.deleteItemsFromArray = exports.addItemsToArray = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _date_interval_utils = require("@kbn/data-plugin/common/search/aggs/utils/date_interval_utils");
var _rule_management = require("../../../../../../common/api/detection_engine/rule_management");
var _invariant = require("../../../../../../common/utils/invariant");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const addItemsToArray = (arr, items) => Array.from(new Set([...arr, ...items]));
exports.addItemsToArray = addItemsToArray;
const deleteItemsFromArray = (arr, items) => {
  const itemsSet = new Set(items);
  return arr.filter(item => !itemsSet.has(item));
};

// Check if current params have a configured data view id
// and the action is not set to overwrite data views
exports.deleteItemsFromArray = deleteItemsFromArray;
const isDataViewExistsAndNotOverriden = (dataViewId, action) => dataViewId != null && !action.overwrite_data_views;

// Check if the index patterns added to the rule already exist in it
const hasIndexPatterns = (indexPatterns, action) => action.value.every(indexPattern => indexPatterns === null || indexPatterns === void 0 ? void 0 : indexPatterns.includes(indexPattern));

// Check if the index patterns to be deleted don't exist in the rule
const hasNotIndexPattern = (indexPatterns, action) => action.value.every(indexPattern => !(indexPatterns !== null && indexPatterns !== void 0 && indexPatterns.includes(indexPattern)));
const shouldSkipIndexPatternsBulkAction = (indexPatterns, dataViewId, action) => {
  if (isDataViewExistsAndNotOverriden(dataViewId, action)) {
    return true;
  }
  if (action.type === _rule_management.BulkActionEditTypeEnum.add_index_patterns) {
    return hasIndexPatterns(indexPatterns, action);
  }
  if (action.type === _rule_management.BulkActionEditTypeEnum.delete_index_patterns) {
    return hasNotIndexPattern(indexPatterns, action);
  }
  return false;
};
const applyBulkActionEditToRuleParams = (existingRuleParams, action) => {
  let ruleParams = {
    ...existingRuleParams
  };
  // If the action is succesfully applied and the rule params are modified,
  // we update the following flag to false. As soon as the current function
  // returns this flag as false, at least once, for any action, we know that
  // the rule needs to be marked as having its params updated.
  let isActionSkipped = false;
  switch (action.type) {
    // index_patterns actions
    // index pattern is not present in machine learning rule type, so we throw error on it
    case _rule_management.BulkActionEditTypeEnum.add_index_patterns:
      {
        var _ruleParams$index;
        (0, _invariant.invariant)(ruleParams.type !== 'machine_learning', "Index patterns can't be added. Machine learning rule doesn't have index patterns property");
        (0, _invariant.invariant)(ruleParams.type !== 'esql', "Index patterns can't be added. ES|QL rule doesn't have index patterns property");
        if (shouldSkipIndexPatternsBulkAction(ruleParams.index, ruleParams.dataViewId, action)) {
          isActionSkipped = true;
          break;
        }
        if (action.overwrite_data_views) {
          ruleParams.dataViewId = undefined;
        }
        ruleParams.index = addItemsToArray((_ruleParams$index = ruleParams.index) !== null && _ruleParams$index !== void 0 ? _ruleParams$index : [], action.value);
        break;
      }
    case _rule_management.BulkActionEditTypeEnum.delete_index_patterns:
      {
        (0, _invariant.invariant)(ruleParams.type !== 'machine_learning', "Index patterns can't be deleted. Machine learning rule doesn't have index patterns property");
        (0, _invariant.invariant)(ruleParams.type !== 'esql', "Index patterns can't be deleted. ES|QL rule doesn't have index patterns property");
        if (!action.overwrite_data_views && shouldSkipIndexPatternsBulkAction(ruleParams.index, ruleParams.dataViewId, action)) {
          isActionSkipped = true;
          break;
        }
        if (action.overwrite_data_views) {
          ruleParams.dataViewId = undefined;
        }
        if (ruleParams.index) {
          ruleParams.index = deleteItemsFromArray(ruleParams.index, action.value);
        }
        break;
      }
    case _rule_management.BulkActionEditTypeEnum.set_index_patterns:
      {
        (0, _invariant.invariant)(ruleParams.type !== 'machine_learning', "Index patterns can't be overwritten. Machine learning rule doesn't have index patterns property");
        (0, _invariant.invariant)(ruleParams.type !== 'esql', "Index patterns can't be overwritten. ES|QL rule doesn't have index patterns property");
        if (shouldSkipIndexPatternsBulkAction(ruleParams.index, ruleParams.dataViewId, action)) {
          isActionSkipped = true;
          break;
        }
        if (action.overwrite_data_views) {
          ruleParams.dataViewId = undefined;
        }
        ruleParams.index = action.value;
        break;
      }
    // timeline actions
    case _rule_management.BulkActionEditTypeEnum.set_timeline:
      {
        ruleParams = {
          ...ruleParams,
          timelineId: action.value.timeline_id || undefined,
          timelineTitle: action.value.timeline_title || undefined
        };
        break;
      }
    // update look-back period in from and meta.from fields
    case _rule_management.BulkActionEditTypeEnum.set_schedule:
      {
        var _parseInterval, _parseInterval2;
        const interval = (_parseInterval = (0, _date_interval_utils.parseInterval)(action.value.interval)) !== null && _parseInterval !== void 0 ? _parseInterval : _moment.default.duration(0);
        const parsedFrom = (_parseInterval2 = (0, _date_interval_utils.parseInterval)(action.value.lookback)) !== null && _parseInterval2 !== void 0 ? _parseInterval2 : _moment.default.duration(0);
        const from = parsedFrom.asSeconds() + interval.asSeconds();
        ruleParams = {
          ...ruleParams,
          meta: {
            ...ruleParams.meta,
            from: action.value.lookback
          },
          from: `now-${from}s`
        };
        break;
      }
  }
  return {
    ruleParams,
    isActionSkipped
  };
};

/**
 * takes list of bulkEdit actions and apply them to rule.params by mutating it
 * @param existingRuleParams
 * @param actions
 * @returns mutated params, isParamsUpdateSkipped flag
 */
const ruleParamsModifier = (existingRuleParams, actions) => {
  let isParamsUpdateSkipped = true;
  const modifiedParams = actions.reduce((acc, action) => {
    const {
      ruleParams,
      isActionSkipped
    } = applyBulkActionEditToRuleParams(acc, action);

    // The rule was updated with at least one action, so mark our rule as updated
    if (!isActionSkipped) {
      isParamsUpdateSkipped = false;
    }
    return {
      ...acc,
      ...ruleParams
    };
  }, existingRuleParams);

  // increment version even if actions are empty, as attributes can be modified as well outside of ruleParamsModifier
  // version must not be modified for immutable rule. Otherwise prebuilt rules upgrade flow will be broken
  if (existingRuleParams.immutable === false) {
    modifiedParams.version += 1;
  }
  return {
    modifiedParams,
    isParamsUpdateSkipped
  };
};
exports.ruleParamsModifier = ruleParamsModifier;