"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createTransform = createTransform;
exports.deleteTransforms = deleteTransforms;
exports.getTransformState = getTransformState;
exports.getTransformsState = getTransformsState;
exports.startTransforms = startTransforms;
exports.stopTransforms = stopTransforms;
var _translations = require("./translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TRANSFORM_API_BASE_PATH = `/internal/transform`;
const toastLifeTimeMs = 600000;
const getErrorToastMessage = ({
  messageBody,
  renderDocLink
}) => renderDocLink ? renderDocLink(messageBody) : messageBody;
async function createTransform({
  errorMessage,
  http,
  notifications,
  options,
  renderDocLink,
  signal,
  transformId
}) {
  const res = await http.put(`${TRANSFORM_API_BASE_PATH}/transforms/${transformId}`, {
    body: JSON.stringify(options),
    version: '1',
    signal
  }).then(result => {
    const {
      errors
    } = result;
    const errorMessageTitle = errorMessage !== null && errorMessage !== void 0 ? errorMessage : _translations.TRANSFORM_CREATION_ERROR_MESSAGE;
    if (errors && errors.length > 0) {
      var _notifications$toasts;
      const failedIds = errors === null || errors === void 0 ? void 0 : errors.map(({
        id,
        error
      }) => {
        var _error$output, _error$output$payload;
        if (error !== null && error !== void 0 && (_error$output = error.output) !== null && _error$output !== void 0 && (_error$output$payload = _error$output.payload) !== null && _error$output$payload !== void 0 && _error$output$payload.cause) {
          var _error$output2, _error$output2$payloa;
          return `${id}: ${error === null || error === void 0 ? void 0 : (_error$output2 = error.output) === null || _error$output2 === void 0 ? void 0 : (_error$output2$payloa = _error$output2.payload) === null || _error$output2$payloa === void 0 ? void 0 : _error$output2$payloa.cause}`;
        }
        return id;
      }, []);
      notifications === null || notifications === void 0 ? void 0 : (_notifications$toasts = notifications.toasts) === null || _notifications$toasts === void 0 ? void 0 : _notifications$toasts.addError(new Error(errorMessageTitle), {
        title: errorMessageTitle,
        toastMessage: getErrorToastMessage({
          messageBody: failedIds.join(', '),
          renderDocLink
        }),
        toastLifeTimeMs
      });
    }
    return result;
  }).catch(e => {
    var _notifications$toasts2, _e$body;
    notifications === null || notifications === void 0 ? void 0 : (_notifications$toasts2 = notifications.toasts) === null || _notifications$toasts2 === void 0 ? void 0 : _notifications$toasts2.addError(e, {
      title: errorMessage !== null && errorMessage !== void 0 ? errorMessage : _translations.TRANSFORM_CREATION_ERROR_MESSAGE,
      toastMessage: getErrorToastMessage({
        messageBody: e === null || e === void 0 ? void 0 : (_e$body = e.body) === null || _e$body === void 0 ? void 0 : _e$body.message,
        renderDocLink
      }),
      toastLifeTimeMs
    });
  });
  return res;
}
async function startTransforms({
  http,
  notifications,
  renderDocLink,
  signal,
  errorMessage,
  transformIds
}) {
  const res = await http.post(`${TRANSFORM_API_BASE_PATH}/start_transforms`, {
    body: JSON.stringify(transformIds.map(id => ({
      id
    }))),
    version: '1',
    signal
  }).then(result => {
    const failedIds = Object.entries(result).reduce((acc, [key, val]) => {
      var _val$error, _val$error2;
      return !val.success ? [...acc, val !== null && val !== void 0 && (_val$error = val.error) !== null && _val$error !== void 0 && _val$error.reason ? `${key}: ${val === null || val === void 0 ? void 0 : (_val$error2 = val.error) === null || _val$error2 === void 0 ? void 0 : _val$error2.reason}` : key] : acc;
    }, []);
    const errorMessageTitle = errorMessage !== null && errorMessage !== void 0 ? errorMessage : (0, _translations.START_TRANSFORMS_ERROR_MESSAGE)(failedIds.length);
    if (failedIds.length > 0) {
      var _notifications$toasts3;
      notifications === null || notifications === void 0 ? void 0 : (_notifications$toasts3 = notifications.toasts) === null || _notifications$toasts3 === void 0 ? void 0 : _notifications$toasts3.addError(new Error(errorMessageTitle), {
        title: errorMessageTitle,
        toastMessage: getErrorToastMessage({
          messageBody: failedIds.join(', '),
          renderDocLink
        }),
        toastLifeTimeMs
      });
    }
    return result;
  }).catch(e => {
    var _notifications$toasts4, _e$body2;
    notifications === null || notifications === void 0 ? void 0 : (_notifications$toasts4 = notifications.toasts) === null || _notifications$toasts4 === void 0 ? void 0 : _notifications$toasts4.addError(e, {
      title: errorMessage !== null && errorMessage !== void 0 ? errorMessage : (0, _translations.START_TRANSFORMS_ERROR_MESSAGE)(transformIds.length),
      toastMessage: getErrorToastMessage({
        messageBody: e === null || e === void 0 ? void 0 : (_e$body2 = e.body) === null || _e$body2 === void 0 ? void 0 : _e$body2.message,
        renderDocLink
      }),
      toastLifeTimeMs
    });
  });
  return res;
}
async function getTransformState({
  http,
  notifications,
  renderDocLink,
  signal,
  errorMessage = _translations.GET_TRANSFORM_STATE_ERROR_MESSAGE,
  transformId
}) {
  const res = await http.get(`${TRANSFORM_API_BASE_PATH}/transforms/${transformId}/_stats`, {
    version: '1',
    signal
  }).then(result => {
    if (result.count === 0) {
      var _notifications$toasts5;
      notifications === null || notifications === void 0 ? void 0 : (_notifications$toasts5 = notifications.toasts) === null || _notifications$toasts5 === void 0 ? void 0 : _notifications$toasts5.addError(new Error(errorMessage), {
        title: errorMessage,
        toastMessage: getErrorToastMessage({
          messageBody: `${_translations.GET_TRANSFORM_STATE_NOT_FOUND_MESSAGE}: ${transformId}`,
          renderDocLink
        }),
        toastLifeTimeMs
      });
    }
    return result;
  }).catch(e => {
    var _notifications$toasts6, _e$body3;
    notifications === null || notifications === void 0 ? void 0 : (_notifications$toasts6 = notifications.toasts) === null || _notifications$toasts6 === void 0 ? void 0 : _notifications$toasts6.addError(e, {
      title: errorMessage,
      toastMessage: getErrorToastMessage({
        messageBody: e === null || e === void 0 ? void 0 : (_e$body3 = e.body) === null || _e$body3 === void 0 ? void 0 : _e$body3.message,
        renderDocLink
      }),
      toastLifeTimeMs
    });
  });
  return res;
}
async function getTransformsState({
  http,
  notifications,
  signal,
  errorMessage,
  transformIds
}) {
  const states = await Promise.all(transformIds.map(transformId => {
    const transformState = getTransformState({
      http,
      notifications,
      signal,
      errorMessage,
      transformId
    });
    return transformState;
  }));
  return states;
}
async function stopTransforms({
  http,
  notifications,
  signal,
  errorMessage,
  transformIds,
  renderDocLink
}) {
  const states = await getTransformsState({
    http,
    signal,
    transformIds
  });
  const res = await http.post(`${TRANSFORM_API_BASE_PATH}/stop_transforms`, {
    version: '1',
    body: JSON.stringify(states.reduce((acc, state) => state != null && state.transforms.length > 0 ? [...acc, {
      id: state.transforms[0].id,
      state: state.transforms[0].state
    }] : acc, [])),
    signal
  }).then(result => {
    const failedIds = Object.entries(result).reduce((acc, [key, val]) => {
      var _val$error3, _val$error4;
      return !val.success ? [...acc, val !== null && val !== void 0 && (_val$error3 = val.error) !== null && _val$error3 !== void 0 && _val$error3.reason ? `${key}: ${val === null || val === void 0 ? void 0 : (_val$error4 = val.error) === null || _val$error4 === void 0 ? void 0 : _val$error4.reason}` : key] : acc;
    }, []);
    const errorMessageTitle = errorMessage !== null && errorMessage !== void 0 ? errorMessage : (0, _translations.STOP_TRANSFORMS_ERROR_MESSAGE)(failedIds.length);
    if (failedIds.length > 0) {
      var _notifications$toasts7;
      notifications === null || notifications === void 0 ? void 0 : (_notifications$toasts7 = notifications.toasts) === null || _notifications$toasts7 === void 0 ? void 0 : _notifications$toasts7.addError(new Error(errorMessageTitle), {
        title: errorMessageTitle,
        toastMessage: getErrorToastMessage({
          messageBody: failedIds.join(', '),
          renderDocLink
        }),
        toastLifeTimeMs
      });
    }
    return result;
  }).catch(e => {
    var _notifications$toasts8, _e$body4;
    notifications === null || notifications === void 0 ? void 0 : (_notifications$toasts8 = notifications.toasts) === null || _notifications$toasts8 === void 0 ? void 0 : _notifications$toasts8.addError(e, {
      title: errorMessage !== null && errorMessage !== void 0 ? errorMessage : (0, _translations.STOP_TRANSFORMS_ERROR_MESSAGE)(transformIds.length),
      toastMessage: getErrorToastMessage({
        messageBody: e === null || e === void 0 ? void 0 : (_e$body4 = e.body) === null || _e$body4 === void 0 ? void 0 : _e$body4.message,
        renderDocLink
      }),
      toastLifeTimeMs
    });
  });
  return res;
}
async function deleteTransforms({
  http,
  notifications,
  signal,
  errorMessage,
  transformIds,
  options,
  renderDocLink
}) {
  await stopTransforms({
    http,
    signal,
    transformIds
  });
  const res = await http.post(`${TRANSFORM_API_BASE_PATH}/delete_transforms`, {
    version: '1',
    body: JSON.stringify({
      transformsInfo: transformIds.map(id => ({
        id,
        state: 'stopped'
      })),
      ...(options ? options : {})
    }),
    signal
  }).then(result => {
    const failedIds = Object.entries(result).reduce((acc, [key, val]) => {
      var _val$transformDeleted, _val$transformDeleted2, _val$transformDeleted3, _val$transformDeleted4;
      return !val.transformDeleted.success ? [...acc, val !== null && val !== void 0 && (_val$transformDeleted = val.transformDeleted) !== null && _val$transformDeleted !== void 0 && (_val$transformDeleted2 = _val$transformDeleted.error) !== null && _val$transformDeleted2 !== void 0 && _val$transformDeleted2.reason ? `${key}: ${val === null || val === void 0 ? void 0 : (_val$transformDeleted3 = val.transformDeleted) === null || _val$transformDeleted3 === void 0 ? void 0 : (_val$transformDeleted4 = _val$transformDeleted3.error) === null || _val$transformDeleted4 === void 0 ? void 0 : _val$transformDeleted4.reason}` : key] : acc;
    }, []);
    const errorMessageTitle = errorMessage !== null && errorMessage !== void 0 ? errorMessage : (0, _translations.TRANSFORM_DELETION_ERROR_MESSAGE)(failedIds.length);
    if (failedIds.length > 0) {
      var _notifications$toasts9;
      notifications === null || notifications === void 0 ? void 0 : (_notifications$toasts9 = notifications.toasts) === null || _notifications$toasts9 === void 0 ? void 0 : _notifications$toasts9.addError(new Error(errorMessageTitle), {
        title: errorMessageTitle,
        toastMessage: getErrorToastMessage({
          messageBody: failedIds.join(', '),
          renderDocLink
        }),
        toastLifeTimeMs
      });
    }
    return result;
  }).catch(e => {
    var _notifications$toasts10, _e$body5;
    notifications === null || notifications === void 0 ? void 0 : (_notifications$toasts10 = notifications.toasts) === null || _notifications$toasts10 === void 0 ? void 0 : _notifications$toasts10.addError(e, {
      title: errorMessage !== null && errorMessage !== void 0 ? errorMessage : (0, _translations.TRANSFORM_DELETION_ERROR_MESSAGE)(transformIds.length),
      toastMessage: getErrorToastMessage({
        messageBody: e === null || e === void 0 ? void 0 : (_e$body5 = e.body) === null || _e$body5 === void 0 ? void 0 : _e$body5.message,
        renderDocLink
      }),
      toastLifeTimeMs
    });
  });
  return res;
}