"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateIntegrationDetails = void 0;
var _lodash = require("lodash");
var _semver = _interopRequireDefault(require("semver"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Given an array of integrations and an array of installed integrations this will return an
 * array of integrations augmented with install details like targetVersion, and `version_satisfied`
 * has.
 */
const calculateIntegrationDetails = (relatedIntegrations, installedIntegrations) => {
  const integrationMatches = findIntegrationMatches(relatedIntegrations, installedIntegrations);
  const integrationDetails = integrationMatches.map(integration => {
    return createIntegrationDetails(integration);
  });
  return integrationDetails.sort((a, b) => {
    return a.integrationTitle.localeCompare(b.integrationTitle);
  });
};
exports.calculateIntegrationDetails = calculateIntegrationDetails;
const findIntegrationMatches = (relatedIntegrations, installedIntegrations) => {
  return relatedIntegrations.map(ri => {
    if (installedIntegrations == null) {
      return {
        related: ri,
        installed: null,
        isLoaded: false
      };
    } else {
      const match = installedIntegrations.find(ii => ii.package_name === ri.package && (ii === null || ii === void 0 ? void 0 : ii.integration_name) === (ri === null || ri === void 0 ? void 0 : ri.integration));
      return {
        related: ri,
        installed: match !== null && match !== void 0 ? match : null,
        isLoaded: true
      };
    }
  });
};
const createIntegrationDetails = integration => {
  var _related$integration;
  const {
    related,
    installed,
    isLoaded
  } = integration;
  const packageName = related.package;
  const integrationName = (_related$integration = related.integration) !== null && _related$integration !== void 0 ? _related$integration : null;
  const requiredVersion = related.version;

  // We don't know whether the integration is installed or not.
  if (!isLoaded) {
    const integrationTitle = getCapitalizedTitle(packageName, integrationName);
    const targetVersion = getMinimumConcreteVersionMatchingSemver(requiredVersion);
    const targetUrl = buildTargetUrl(packageName, integrationName, targetVersion);
    return {
      packageName,
      integrationName,
      integrationTitle,
      requiredVersion,
      targetVersion,
      targetUrl,
      installationStatus: {
        isKnown: false
      }
    };
  }

  // We know that the integration is not installed
  if (installed == null) {
    const integrationTitle = getCapitalizedTitle(packageName, integrationName);
    const targetVersion = getMinimumConcreteVersionMatchingSemver(requiredVersion);
    const targetUrl = buildTargetUrl(packageName, integrationName, targetVersion);
    return {
      packageName,
      integrationName,
      integrationTitle,
      requiredVersion,
      targetVersion,
      targetUrl,
      installationStatus: {
        isKnown: true,
        isInstalled: false,
        isEnabled: false,
        isVersionMismatch: false,
        installedVersion: ''
      }
    };
  }

  // We know that the integration is installed
  {
    var _installed$integratio;
    const integrationTitle = (_installed$integratio = installed.integration_title) !== null && _installed$integratio !== void 0 ? _installed$integratio : installed.package_title;

    // Version check e.g. installed version `1.2.3` satisfies required version `~1.2.1`
    const installedVersion = installed.package_version;
    const isVersionSatisfied = _semver.default.satisfies(installedVersion, requiredVersion);
    const targetVersion = isVersionSatisfied ? installedVersion : getMinimumConcreteVersionMatchingSemver(requiredVersion);
    const targetUrl = buildTargetUrl(packageName, integrationName, targetVersion);
    return {
      packageName,
      integrationName,
      integrationTitle,
      requiredVersion,
      targetVersion,
      targetUrl,
      installationStatus: {
        isKnown: true,
        isInstalled: true,
        isEnabled: installed.is_enabled,
        isVersionMismatch: !isVersionSatisfied,
        installedVersion
      }
    };
  }
};
const getCapitalizedTitle = (packageName, integrationName) => {
  return integrationName == null ? `${(0, _lodash.capitalize)(packageName)}` : `${(0, _lodash.capitalize)(packageName)} ${(0, _lodash.capitalize)(integrationName)}`;
};
const getMinimumConcreteVersionMatchingSemver = semverString => {
  var _semver$valid;
  return (_semver$valid = _semver.default.valid(_semver.default.coerce(semverString))) !== null && _semver$valid !== void 0 ? _semver$valid : '';
};
const buildTargetUrl = (packageName, integrationName, targetVersion) => {
  const packageSegment = targetVersion ? `${packageName}-${targetVersion}` : packageName;
  const query = integrationName ? `?integration=${integrationName}` : '';
  return `app/integrations/detail/${packageSegment}/overview${query}`;
};