"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDiscoverInTimelineActions = exports.defaultDiscoverTimeRange = void 0;
var _lodash = require("lodash");
var _react = require("react");
var _reactRedux = require("react-redux");
var _reactQuery = require("@tanstack/react-query");
var _defaults = require("../../../timelines/store/timeline/defaults");
var _types = require("../../../../common/types");
var _timeline = require("../../../timelines/store/timeline");
var _use_app_toasts = require("../../hooks/use_app_toasts");
var _use_selector = require("../../hooks/use_selector");
var _kibana = require("../../lib/kibana");
var _utils = require("../../../timelines/components/timeline/esql_tab_content/utils");
var _translations = require("./translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const defaultDiscoverTimeRange = {
  from: 'now-15m',
  to: 'now',
  mode: 'relative'
};
exports.defaultDiscoverTimeRange = defaultDiscoverTimeRange;
const useDiscoverInTimelineActions = discoverStateContainer => {
  const {
    addError
  } = (0, _use_app_toasts.useAppToasts)();
  const {
    services: {
      customDataService: discoverDataService,
      savedSearch: savedSearchService
    }
  } = (0, _kibana.useKibana)();
  const dispatch = (0, _reactRedux.useDispatch)();
  const getTimeline = (0, _react.useMemo)(() => _timeline.timelineSelectors.getTimelineByIdSelector(), []);
  const timeline = (0, _use_selector.useShallowEqualSelector)(state => {
    var _getTimeline;
    return (_getTimeline = getTimeline(state, _types.TimelineId.active)) !== null && _getTimeline !== void 0 ? _getTimeline : _defaults.timelineDefaults;
  });
  const {
    savedSearchId,
    version
  } = timeline;

  // We're using a ref here to prevent a cyclic hook-dependency chain of updateSavedSearch
  const timelineRef = (0, _react.useRef)(timeline);
  timelineRef.current = timeline;
  const queryClient = (0, _reactQuery.useQueryClient)();
  const {
    mutateAsync: saveSavedSearch,
    status
  } = (0, _reactQuery.useMutation)({
    mutationFn: ({
      savedSearch,
      savedSearchOptions
    }) => savedSearchService.save(savedSearch, savedSearchOptions),
    onSuccess: data => {
      // Invalidate and refetch
      if (data) {
        dispatch(_timeline.timelineActions.endTimelineSaving({
          id: _types.TimelineId.active
        }));
      }
      queryClient.invalidateQueries({
        queryKey: ['savedSearchById', savedSearchId]
      });
    },
    mutationKey: [version]
  });
  const getDefaultDiscoverAppState = (0, _react.useCallback)(async () => {
    return {
      query: {
        esql: ''
      },
      sort: [['@timestamp', 'desc']],
      columns: [],
      interval: 'auto',
      filters: [],
      hideChart: true,
      grid: {}
    };
  }, []);

  /*
   * generates Appstate from a given saved Search object
   *
   * @param savedSearch
   *
   * */
  const getAppStateFromSavedSearch = (0, _react.useCallback)(savedSearch => {
    var _discoverStateContain;
    const appState = (_discoverStateContain = discoverStateContainer.current) === null || _discoverStateContain === void 0 ? void 0 : _discoverStateContain.appState.getAppStateFromSavedSearch(savedSearch);
    return {
      savedSearch,
      appState
    };
  }, [discoverStateContainer]);

  /*
   * restores the url state of discover in timeline
   *
   * @param savedSearch
   * */
  const restoreDiscoverAppStateFromSavedSearch = (0, _react.useCallback)(savedSearch => {
    var _discoverStateContain2, _discoverStateContain3, _discoverStateContain4;
    const {
      appState
    } = getAppStateFromSavedSearch(savedSearch);
    if (!appState) return;
    (_discoverStateContain2 = discoverStateContainer.current) === null || _discoverStateContain2 === void 0 ? void 0 : _discoverStateContain2.appState.set(appState);
    const timeRangeFromSavedSearch = savedSearch.timeRange;
    (_discoverStateContain3 = discoverStateContainer.current) === null || _discoverStateContain3 === void 0 ? void 0 : _discoverStateContain3.globalState.set({
      ...((_discoverStateContain4 = discoverStateContainer.current) === null || _discoverStateContain4 === void 0 ? void 0 : _discoverStateContain4.globalState.get()),
      time: timeRangeFromSavedSearch !== null && timeRangeFromSavedSearch !== void 0 ? timeRangeFromSavedSearch : defaultDiscoverTimeRange
    });
  }, [getAppStateFromSavedSearch, discoverStateContainer]);

  /*
   * resets discover state to a default value
   *
   * */
  const resetDiscoverAppState = (0, _react.useCallback)(async () => {
    var _discoverStateContain5, _discoverStateContain6, _discoverStateContain7;
    const defaultDiscoverAppState = await getDefaultDiscoverAppState();
    (_discoverStateContain5 = discoverStateContainer.current) === null || _discoverStateContain5 === void 0 ? void 0 : _discoverStateContain5.appState.replaceUrlState(defaultDiscoverAppState);
    (_discoverStateContain6 = discoverStateContainer.current) === null || _discoverStateContain6 === void 0 ? void 0 : _discoverStateContain6.globalState.set({
      ...((_discoverStateContain7 = discoverStateContainer.current) === null || _discoverStateContain7 === void 0 ? void 0 : _discoverStateContain7.globalState.get()),
      time: defaultDiscoverTimeRange
    });
  }, [getDefaultDiscoverAppState, discoverStateContainer]);
  const persistSavedSearch = (0, _react.useCallback)(async (savedSearch, savedSearchOption) => {
    if (!discoverStateContainer) {
      // eslint-disable-next-line no-console
      console.log(`Saved search is not open since state container is null`);
      return;
    }
    if (!savedSearch) return;
    function onError(error) {
      addError(error, {
        title: _translations.DISCOVER_SEARCH_SAVE_ERROR_TITLE
      });
    }
    try {
      const id = await saveSavedSearch({
        savedSearch,
        savedSearchOptions: savedSearchOption
      });
      if (id) {
        return {
          id
        };
      } else {
        addError(_translations.DISCOVER_SEARCH_SAVE_ERROR_UNKNOWN, {
          title: _translations.DISCOVER_SEARCH_SAVE_ERROR_TITLE
        });
      }
    } catch (err) {
      onError(err);
    }
  }, [addError, discoverStateContainer, saveSavedSearch]);

  /*
   * persists the given savedSearch
   *
   * */
  const updateSavedSearch = (0, _react.useCallback)(async (savedSearch, timelineId) => {
    var _savedSearch$timeRang;
    savedSearch.timeRestore = true;
    savedSearch.timeRange = (_savedSearch$timeRang = savedSearch.timeRange) !== null && _savedSearch$timeRang !== void 0 ? _savedSearch$timeRang : discoverDataService.query.timefilter.timefilter.getTime();
    savedSearch.tags = ['security-solution-default'];

    // If there is already a saved search, only update the local state
    if (savedSearchId) {
      savedSearch.id = savedSearchId;
      if (!timelineRef.current.savedSearch) {
        dispatch(_timeline.timelineActions.initializeSavedSearch({
          id: _types.TimelineId.active,
          savedSearch
        }));
      } else if (!(0, _lodash.isEqualWith)(timelineRef.current.savedSearch, savedSearch, _utils.savedSearchComparator)) {
        dispatch(_timeline.timelineActions.updateSavedSearch({
          id: _types.TimelineId.active,
          savedSearch
        }));
      }
    } else {
      // If no saved search exists. Create a new saved search instance and associate it with the timeline.
      try {
        // Make sure we're not creating a saved search while a previous creation call is in progress
        if (status !== 'idle') {
          return;
        }
        dispatch(_timeline.timelineActions.startTimelineSaving({
          id: _types.TimelineId.active
        }));
        const response = await persistSavedSearch(savedSearch, {
          onTitleDuplicate: () => {},
          copyOnSave: !savedSearchId
        });
        const responseIsEmpty = !response || !(response !== null && response !== void 0 && response.id);
        if (responseIsEmpty) {
          throw new Error('Response is empty');
        } else if (!savedSearchId && !responseIsEmpty) {
          dispatch(_timeline.timelineActions.updateSavedSearchId({
            id: _types.TimelineId.active,
            savedSearchId: response.id
          }));
          // Also save the timeline, this will only happen once, in case there is no saved search id yet
          dispatch(_timeline.timelineActions.saveTimeline({
            id: _types.TimelineId.active,
            saveAsNew: false
          }));
        }
      } catch (err) {
        dispatch(_timeline.timelineActions.endTimelineSaving({
          id: _types.TimelineId.active
        }));
      }
    }
  }, [persistSavedSearch, savedSearchId, dispatch, discoverDataService, status]);
  const initializeLocalSavedSearch = (0, _react.useCallback)(async (savedSearch, timelineId) => {
    dispatch(_timeline.timelineActions.initializeSavedSearch({
      id: _types.TimelineId.active,
      savedSearch
    }));
  }, [dispatch]);
  const actions = (0, _react.useMemo)(() => ({
    resetDiscoverAppState,
    restoreDiscoverAppStateFromSavedSearch,
    updateSavedSearch,
    initializeLocalSavedSearch,
    getAppStateFromSavedSearch,
    getDefaultDiscoverAppState
  }), [resetDiscoverAppState, restoreDiscoverAppStateFromSavedSearch, updateSavedSearch, initializeLocalSavedSearch, getAppStateFromSavedSearch, getDefaultDiscoverAppState]);
  return actions;
};
exports.useDiscoverInTimelineActions = useDiscoverInTimelineActions;