"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.withLink = exports.useGetLinkUrl = exports.useGetLinkProps = exports.isSecurityId = exports.isModified = exports.getAppIdsFromId = exports.formatPath = exports.concatPaths = exports.LinkButton = exports.LinkAnchor = void 0;
var _extends2 = _interopRequireDefault(require("@babel/runtime/helpers/extends"));
var _react = _interopRequireWildcard(require("react"));
var _eui = require("@elastic/eui");
var _navigation = require("./navigation");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * It returns the `url` to use in link `href`.
 */
const useGetLinkUrl = () => {
  const {
    getAppUrl
  } = (0, _navigation.useGetAppUrl)();
  const getLinkUrl = (0, _react.useCallback)(({
    id,
    path: subPath = '',
    absolute = false,
    urlState
  }) => {
    const {
      appId,
      deepLinkId,
      path: mainPath = ''
    } = getAppIdsFromId(id);
    const path = concatPaths(mainPath, subPath);
    const formattedPath = urlState ? formatPath(path, urlState) : path;
    return getAppUrl({
      deepLinkId,
      appId,
      path: formattedPath,
      absolute
    });
  }, [getAppUrl]);
  return getLinkUrl;
};

/**
 * It returns the `onClick` and `href` props to use in link components based on the` deepLinkId` and `path` parameters.
 */
exports.useGetLinkUrl = useGetLinkUrl;
const useGetLinkProps = () => {
  const getLinkUrl = useGetLinkUrl();
  const {
    navigateTo
  } = (0, _navigation.useNavigateTo)();
  const getLinkProps = (0, _react.useCallback)(({
    id,
    path,
    urlState,
    onClick: onClickProps,
    overrideNavigation = true
  }) => {
    const url = getLinkUrl({
      id,
      path,
      urlState
    });
    return {
      href: url,
      onClick: ev => {
        if (isModified(ev)) {
          return;
        }
        if (onClickProps) {
          onClickProps(ev);
        }
        if (overrideNavigation) {
          ev.preventDefault();
          navigateTo({
            url
          });
        }
      }
    };
  }, [getLinkUrl, navigateTo]);
  return getLinkProps;
};

/**
 * HOC that wraps any Link component and makes it a navigation Link.
 */
exports.useGetLinkProps = useGetLinkProps;
const withLink = Component => /*#__PURE__*/_react.default.memo(function WithLink({
  id,
  path,
  urlState,
  onClick: _onClick,
  ...rest
}) {
  const getLink = useGetLinkProps();
  const {
    onClick,
    href
  } = getLink({
    id,
    path,
    urlState,
    onClick: _onClick,
    ...(rest.target === '_blank' && {
      overrideNavigation: false
    })
  });
  return /*#__PURE__*/_react.default.createElement(Component, (0, _extends2.default)({
    onClick: onClick,
    href: href
  }, rest));
});

/**
 * Security Solutions internal link button.
 *
 * `<LinkButton deepLinkId={SecurityPageName.hosts} />;`
 */
exports.withLink = withLink;
const LinkButton = withLink(_eui.EuiButton);

/**
 * Security Solutions internal link anchor.
 *
 * `<LinkAnchor deepLinkId={SecurityPageName.hosts} />;`
 */
exports.LinkButton = LinkButton;
const LinkAnchor = withLink(_eui.EuiLink);

// Utils

// External IDs are in the format `appId:deepLinkId` to match the Chrome NavLinks format.
// Internal Security Solution ids are the deepLinkId, the appId is omitted for convenience.
exports.LinkAnchor = LinkAnchor;
const isSecurityId = id => !id.includes(':');

// External links may contain an optional `path` in addition to the `appId` and `deepLinkId`.
// Format: `<appId>:<deepLinkId>/<path>`
exports.isSecurityId = isSecurityId;
const getAppIdsFromId = id => {
  const [linkId, strippedPath] = id.split(/\/(.*)/); // split by the first `/` character
  const path = strippedPath ? `/${strippedPath}` : '';
  if (!isSecurityId(linkId)) {
    const [appId, deepLinkId] = linkId.split(':');
    return {
      appId,
      deepLinkId,
      path
    };
  }
  return {
    deepLinkId: linkId,
    path
  }; // undefined `appId` for internal Security Solution links
};
exports.getAppIdsFromId = getAppIdsFromId;
const concatPaths = (path, subPath) => {
  if (path && subPath) {
    return `${path.replace(/\/$/, '')}/${subPath.replace(/^\//, '')}`;
  }
  return path || subPath || '';
};
exports.concatPaths = concatPaths;
const formatPath = (path, urlState) => {
  const urlStateClean = urlState.replace('?', '');
  const [urlPath, parameterPath] = path.split('?');
  let queryParams = '';
  if (urlStateClean && parameterPath) {
    queryParams = `?${parameterPath}&${urlStateClean}`;
  } else if (parameterPath) {
    queryParams = `?${parameterPath}`;
  } else if (urlStateClean) {
    queryParams = `?${urlStateClean}`;
  }
  return `${urlPath}${queryParams}`;
};
exports.formatPath = formatPath;
const isModified = event => event.metaKey || event.altKey || event.ctrlKey || event.shiftKey;
exports.isModified = isModified;