"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createJob = exports.clearCreateJobErrors = void 0;
var _i18n = require("@kbn/i18n");
var _services = require("../../services");
var _change_job_status = require("./change_job_status");
var _action_types = require("../action_types");
var _kibana_services = require("../../../kibana_services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createJob = jobConfig => async dispatch => {
  dispatch({
    type: _action_types.CREATE_JOB_START
  });
  let newJob;
  try {
    [newJob] = await Promise.all([(0, _services.createJob)((0, _services.serializeJob)(jobConfig)),
    // Wait at least half a second to avoid a weird flicker of the saving feedback.
    new Promise(resolve => setTimeout(resolve, 500))]);
  } catch (error) {
    if (error) {
      const {
        body
      } = error;
      const statusCode = error.statusCode || body && body.statusCode;

      // Expect an error in the shape provided by http service.
      if (body) {
        // Some errors have statusCode directly available but some are under a data property.
        if (statusCode === 409) {
          return dispatch({
            type: _action_types.CREATE_JOB_FAILURE,
            payload: {
              error: {
                message: _i18n.i18n.translate('xpack.rollupJobs.createAction.jobIdAlreadyExistsErrorMessage', {
                  defaultMessage: `A job with ID '{jobConfigId}' already exists.`,
                  values: {
                    jobConfigId: jobConfig.id
                  }
                })
              }
            }
          });
        }
        return dispatch({
          type: _action_types.CREATE_JOB_FAILURE,
          payload: {
            error: {
              message: _i18n.i18n.translate('xpack.rollupJobs.createAction.failedDefaultErrorMessage', {
                defaultMessage: 'Request failed with a {statusCode} error. {message}',
                values: {
                  statusCode,
                  message: body.message
                }
              }),
              cause: body.cause
            }
          }
        });
      }
    }

    // This error isn't an HTTP error, so let the fatal error screen tell the user something
    // unexpected happened.
    return (0, _kibana_services.getFatalErrors)().add(error, _i18n.i18n.translate('xpack.rollupJobs.createAction.errorTitle', {
      defaultMessage: 'Error creating rollup job'
    }));
  }
  const deserializedJob = (0, _services.deserializeJob)(newJob);
  dispatch({
    type: _action_types.CREATE_JOB_SUCCESS,
    payload: {
      job: deserializedJob
    }
  });
  if (jobConfig.startJobAfterCreation) {
    dispatch((0, _change_job_status.startJobs)([jobConfig.id]));
  }

  // This will open the new job in the detail panel. Note that we're *not* showing a success toast
  // here, because it would partially obscure the detail panel.
  (0, _services.getRouter)().history.push({
    pathname: `/job_list`,
    search: `?job=${encodeURIComponent(jobConfig.id)}`
  });
};
exports.createJob = createJob;
const clearCreateJobErrors = () => dispatch => {
  dispatch({
    type: _action_types.CLEAR_CREATE_JOB_ERRORS
  });
};
exports.clearCreateJobErrors = clearCreateJobErrors;