"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.removeClusters = void 0;
var _i18n = require("@kbn/i18n");
var _notification = require("../../services/notification");
var _constants = require("../../constants");
var _services = require("../../services");
var _action_types = require("../action_types");
var _detail_panel = require("./detail_panel");
var _selectors = require("../selectors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getErrorTitle(count, name = null) {
  if (count === 1) {
    if (name) {
      return _i18n.i18n.translate('xpack.remoteClusters.removeAction.errorSingleNotificationTitle', {
        defaultMessage: `Error removing remote cluster '{name}'`,
        values: {
          name
        }
      });
    }
  } else {
    return _i18n.i18n.translate('xpack.remoteClusters.removeAction.errorMultipleNotificationTitle', {
      defaultMessage: `Error removing {count} remote clusters`,
      values: {
        count
      }
    });
  }
}
const removeClusters = names => async (dispatch, getState) => {
  dispatch({
    type: _action_types.REMOVE_CLUSTERS_START
  });
  let itemsDeleted = [];
  let errors = [];
  await Promise.all([(0, _services.removeClusterRequest)(names.join(',')).then(response => {
    ({
      itemsDeleted,
      errors
    } = response);
  }),
  // Wait at least half a second to avoid a weird flicker of the saving feedback (only visible
  // when requests resolve very quickly).
  new Promise(resolve => setTimeout(resolve, 500))]).catch(error => {
    var _error$body;
    const errorTitle = getErrorTitle(names.length, names[0]);
    _notification.toasts.addDanger({
      title: errorTitle,
      text: (_error$body = error.body) === null || _error$body === void 0 ? void 0 : _error$body.message
    });
  });
  if (errors.length > 0) {
    const {
      name,
      error: {
        payload: {
          message
        }
      }
    } = errors[0];
    const title = getErrorTitle(errors.length, name);
    _notification.toasts.addDanger({
      title,
      text: message
    });
  }
  if (itemsDeleted.length > 0) {
    // Only track successful requests.
    (0, _services.trackUiMetric)(_services.METRIC_TYPE.COUNT, names.length > 1 ? _constants.UIM_CLUSTER_REMOVE_MANY : _constants.UIM_CLUSTER_REMOVE);
    if (itemsDeleted.length === 1) {
      _notification.toasts.addSuccess(_i18n.i18n.translate('xpack.remoteClusters.removeAction.successSingleNotificationTitle', {
        defaultMessage: `Remote cluster '{name}' was removed`,
        values: {
          name: itemsDeleted[0]
        }
      }));
    } else {
      _notification.toasts.addSuccess(_i18n.i18n.translate('xpack.remoteClusters.removeAction.successMultipleNotificationTitle', {
        defaultMessage: '{count} remote clusters were removed',
        values: {
          count: itemsDeleted.length
        }
      }));
    }
  }

  // If we've just deleted a cluster we were looking at, we need to close the panel.
  const detailPanelClusterName = (0, _selectors.getDetailPanelClusterName)(getState());
  if (detailPanelClusterName && names.includes(detailPanelClusterName)) {
    dispatch((0, _detail_panel.closeDetailPanel)());
  }
  dispatch({
    type: _action_types.REMOVE_CLUSTERS_FINISH,
    // Send the cluster that have been removed to the reducers
    // and update the store immediately without the need to re-fetch from the server
    payload: itemsDeleted
  });
};
exports.removeClusters = removeClusters;