"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PluginServiceRegistry = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _react = _interopRequireDefault(require("react"));
var _providers_mediator = require("./providers_mediator");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * A `PluginServiceRegistry` maintains a set of service providers which can be collectively
 * started, stopped or retreived.
 *
 * The `Services` generic determines the shape of all service APIs being produced.
 * The `StartParameters` generic determines what parameters are expected to
 * start the service.
 */
class PluginServiceRegistry {
  constructor(providers) {
    (0, _defineProperty2.default)(this, "providers", void 0);
    (0, _defineProperty2.default)(this, "providersMediator", void 0);
    (0, _defineProperty2.default)(this, "_isStarted", false);
    this.providers = providers;
    this.providersMediator = new _providers_mediator.PluginServiceProvidersMediator(providers);
  }

  /**
   * Returns true if the registry has been started, false otherwise.
   */
  isStarted() {
    return this._isStarted;
  }

  /**
   * Returns a map of `PluginServiceProvider` objects.
   */
  getServiceProviders() {
    if (!this._isStarted) {
      throw new Error('Registry not started');
    }
    return this.providers;
  }

  /**
   * Returns a React Context Provider for use in consuming applications.
   */
  getContextProvider() {
    const values = Object.values(this.getServiceProviders());

    // Collect and combine Context.Provider elements from each Service Provider into a single
    // Functional Component.
    const provider = ({
      children
    }) => /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, values.reduceRight((acc, serviceProvider) => {
      return /*#__PURE__*/_react.default.createElement(serviceProvider.Provider, null, acc);
    }, children));
    return provider;
  }

  /**
   * Start the registry.
   *
   * @param params Parameters used to start the registry.
   */
  start(params) {
    this.providersMediator.start(params);
    this._isStarted = true;
    return this;
  }

  /**
   * Stop the registry.
   */
  stop() {
    this.providersMediator.stop();
    this._isStarted = false;
    return this;
  }
}
exports.PluginServiceRegistry = PluginServiceRegistry;