"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefaultSummarySearchClient = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _std = require("@kbn/std");
var _lodash = _interopRequireDefault(require("lodash"));
var _constants = require("../../../common/slo/constants");
var _number = require("../../utils/number");
var _transform_generators = require("./transform_generators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DefaultSummarySearchClient {
  constructor(esClient, logger, spaceId) {
    this.esClient = esClient;
    this.logger = logger;
    this.spaceId = spaceId;
  }
  async search(kqlQuery, sort, pagination) {
    try {
      var _value;
      const summarySearch = await this.esClient.search({
        index: _constants.SLO_SUMMARY_DESTINATION_INDEX_PATTERN,
        track_total_hits: true,
        query: {
          bool: {
            filter: [{
              term: {
                spaceId: this.spaceId
              }
            }, (0, _transform_generators.getElastichsearchQueryOrThrow)(kqlQuery)]
          }
        },
        sort: {
          // non-temp first, then temp documents
          isTempDoc: {
            order: 'asc'
          },
          [toDocumentSortField(sort.field)]: {
            order: sort.direction
          }
        },
        from: (pagination.page - 1) * pagination.perPage,
        size: pagination.perPage * 2 // twice as much as we return, in case they are all duplicate temp/non-temp summary
      });

      const total = (_value = summarySearch.hits.total.value) !== null && _value !== void 0 ? _value : 0;
      if (total === 0) {
        return {
          total: 0,
          perPage: pagination.perPage,
          page: pagination.page,
          results: []
        };
      }
      const [tempSummaryDocuments, summaryDocuments] = _lodash.default.partition(summarySearch.hits.hits, doc => {
        var _doc$_source;
        return !!((_doc$_source = doc._source) !== null && _doc$_source !== void 0 && _doc$_source.isTempDoc);
      });

      // Always attempt to delete temporary summary documents with an existing non-temp summary document
      // The temp summary documents are _eventually_ removed as we get through the real summary documents
      const summarySloIds = summaryDocuments.map(doc => {
        var _doc$_source2;
        return (_doc$_source2 = doc._source) === null || _doc$_source2 === void 0 ? void 0 : _doc$_source2.slo.id;
      });
      await this.esClient.deleteByQuery({
        index: _constants.SLO_SUMMARY_DESTINATION_INDEX_PATTERN,
        wait_for_completion: false,
        query: {
          bool: {
            filter: [{
              terms: {
                'slo.id': summarySloIds
              }
            }, {
              term: {
                isTempDoc: true
              }
            }]
          }
        }
      });
      const tempSummaryDocumentsDeduped = tempSummaryDocuments.filter(doc => {
        var _doc$_source3;
        return !summarySloIds.includes((_doc$_source3 = doc._source) === null || _doc$_source3 === void 0 ? void 0 : _doc$_source3.slo.id);
      });
      const finalResults = summaryDocuments.concat(tempSummaryDocumentsDeduped).slice(0, pagination.perPage);
      const finalTotal = total - (tempSummaryDocuments.length - tempSummaryDocumentsDeduped.length);
      return {
        total: finalTotal,
        perPage: pagination.perPage,
        page: pagination.page,
        results: finalResults.map(doc => {
          var _slo$instanceId;
          return {
            id: doc._source.slo.id,
            instanceId: (_slo$instanceId = doc._source.slo.instanceId) !== null && _slo$instanceId !== void 0 ? _slo$instanceId : _sloSchema.ALL_VALUE,
            summary: {
              errorBudget: {
                initial: (0, _number.toHighPrecision)(doc._source.errorBudgetInitial),
                consumed: (0, _number.toHighPrecision)(doc._source.errorBudgetConsumed),
                remaining: (0, _number.toHighPrecision)(doc._source.errorBudgetRemaining),
                isEstimated: doc._source.errorBudgetEstimated
              },
              sliValue: (0, _number.toHighPrecision)(doc._source.sliValue),
              status: doc._source.status
            }
          };
        })
      };
    } catch (err) {
      this.logger.error(new Error('Summary search query error', {
        cause: err
      }));
      return {
        total: 0,
        perPage: pagination.perPage,
        page: pagination.page,
        results: []
      };
    }
  }
}
exports.DefaultSummarySearchClient = DefaultSummarySearchClient;
function toDocumentSortField(field) {
  switch (field) {
    case 'error_budget_consumed':
      return 'errorBudgetConsumed';
    case 'error_budget_remaining':
      return 'errorBudgetRemaining';
    case 'status':
      return 'status';
    case 'sli_value':
      return 'sliValue';
    default:
      (0, _std.assertNever)(field);
  }
}