"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DeleteSLO = void 0;
var _constants = require("../../../common/slo/constants");
var _retry = require("../../utils/retry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DeleteSLO {
  constructor(repository, transformManager, summaryTransformManager, esClient, rulesClient) {
    this.repository = repository;
    this.transformManager = transformManager;
    this.summaryTransformManager = summaryTransformManager;
    this.esClient = esClient;
    this.rulesClient = rulesClient;
  }
  async execute(sloId) {
    const slo = await this.repository.findById(sloId);
    const summaryTransformId = (0, _constants.getSLOSummaryTransformId)(slo.id, slo.revision);
    await this.summaryTransformManager.stop(summaryTransformId);
    await this.summaryTransformManager.uninstall(summaryTransformId);
    const rollupTransformId = (0, _constants.getSLOTransformId)(slo.id, slo.revision);
    await this.transformManager.stop(rollupTransformId);
    await this.transformManager.uninstall(rollupTransformId);
    await (0, _retry.retryTransientEsErrors)(() => this.esClient.ingest.deletePipeline({
      id: (0, _constants.getSLOSummaryPipelineId)(slo.id, slo.revision)
    }, {
      ignore: [404]
    }));
    await this.deleteRollupData(slo.id);
    await this.deleteSummaryData(slo.id);
    await this.deleteAssociatedRules(slo.id);
    await this.repository.deleteById(slo.id);
  }
  async deleteRollupData(sloId) {
    await this.esClient.deleteByQuery({
      index: _constants.SLO_DESTINATION_INDEX_PATTERN,
      wait_for_completion: false,
      query: {
        match: {
          'slo.id': sloId
        }
      }
    });
  }
  async deleteSummaryData(sloId) {
    await this.esClient.deleteByQuery({
      index: _constants.SLO_SUMMARY_DESTINATION_INDEX_PATTERN,
      refresh: true,
      query: {
        match: {
          'slo.id': sloId
        }
      }
    });
  }
  async deleteAssociatedRules(sloId) {
    try {
      await this.rulesClient.bulkDeleteRules({
        filter: `alert.attributes.params.sloId:${sloId}`
      });
    } catch (err) {
      // no-op: bulkDeleteRules throws if no rules are found.
    }
  }
}
exports.DeleteSLO = DeleteSLO;