"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.evaluate = evaluate;
var _lodash = require("lodash");
var _models = require("../../../../domain/models");
var _constants = require("../../../../../common/slo/constants");
var _build_query = require("./build_query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function queryAllResults(esClient, slo, params, startedAt, buckets = [], lastAfterKey) {
  const queryAndAggs = (0, _build_query.buildQuery)(startedAt, slo, params, lastAfterKey);
  const results = await esClient.search({
    index: _constants.SLO_DESTINATION_INDEX_PATTERN,
    ...queryAndAggs
  });
  if (!results.aggregations) {
    throw new Error('Elasticsearch query failed to return a valid aggregation');
  }
  if (results.aggregations.instances.buckets.length === 0) {
    return buckets;
  }
  return queryAllResults(esClient, slo, params, startedAt, [...buckets, ...results.aggregations.instances.buckets], results.aggregations.instances.after_key);
}
async function evaluate(esClient, slo, params, startedAt) {
  const buckets = await queryAllResults(esClient, slo, params, startedAt);
  return transformBucketToResults(buckets, params);
}
function transformBucketToResults(buckets, params) {
  return buckets.map(bucket => {
    for (const index in params.windows) {
      if (params.windows[index]) {
        const winDef = params.windows[index];
        const windowId = (0, _build_query.generateWindowId)(index);
        const shortWindowThresholdKey = (0, _build_query.generateAboveThresholdKey)(windowId, _build_query.SHORT_WINDOW);
        const longWindowThresholdKey = (0, _build_query.generateAboveThresholdKey)(windowId, _build_query.LONG_WINDOW);
        const isShortWindowTriggering = (0, _lodash.get)(bucket, [shortWindowThresholdKey, 'value'], 0);
        const isLongWindowTriggering = (0, _lodash.get)(bucket, [longWindowThresholdKey, 'value'], 0);
        if (isShortWindowTriggering && isLongWindowTriggering) {
          return {
            instanceId: bucket.key.instanceId,
            shouldAlert: true,
            longWindowBurnRate: (0, _lodash.get)(bucket, [(0, _build_query.generateBurnRateKey)(windowId, _build_query.LONG_WINDOW), 'value'], 0),
            shortWindowBurnRate: (0, _lodash.get)(bucket, [(0, _build_query.generateBurnRateKey)(windowId, _build_query.SHORT_WINDOW), 'value'], 0),
            shortWindowDuration: new _models.Duration(winDef.shortWindow.value, (0, _models.toDurationUnit)(winDef.shortWindow.unit)),
            longWindowDuration: new _models.Duration(winDef.longWindow.value, (0, _models.toDurationUnit)(winDef.longWindow.unit)),
            window: winDef
          };
        }
      }
    }
    throw new Error(`Evaluation query for ${bucket.key.instanceId} failed.`);
  });
}