"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SHORT_WINDOW = exports.LONG_WINDOW = void 0;
exports.buildQuery = buildQuery;
exports.generateAboveThresholdKey = generateAboveThresholdKey;
exports.generateBurnRateKey = generateBurnRateKey;
exports.generateStatsKey = generateStatsKey;
exports.generateWindowId = generateWindowId;
var _sloSchema = require("@kbn/slo-schema");
var _models = require("../../../../domain/models");
var _get_delay_in_seconds_from_slo = require("../../../../domain/services/get_delay_in_seconds_from_slo");
var _get_lookback_date_range = require("../../../../domain/services/get_lookback_date_range");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const LONG_WINDOW = 'LONG';
exports.LONG_WINDOW = LONG_WINDOW;
const SHORT_WINDOW = 'SHORT';
exports.SHORT_WINDOW = SHORT_WINDOW;
const BURN_RATE = 'BURN_RATE';
const WINDOW = 'WINDOW';
const ABOVE_THRESHOLD = 'ABOVE_THRESHOLD';
function generateWindowId(index) {
  return `${WINDOW}_${index}`;
}
function generateStatsKey(id, type) {
  return `${id}_${type}`;
}
function generateBurnRateKey(id, type) {
  return `${generateStatsKey(id, type)}_${BURN_RATE}`;
}
function generateAboveThresholdKey(id, type) {
  return `${generateStatsKey(id, type)}_${ABOVE_THRESHOLD}`;
}
const TIMESLICE_AGGS = {
  good: {
    sum: {
      field: 'slo.isGoodSlice'
    }
  },
  total: {
    value_count: {
      field: 'slo.isGoodSlice'
    }
  }
};
const OCCURRENCE_AGGS = {
  good: {
    sum: {
      field: 'slo.numerator'
    }
  },
  total: {
    sum: {
      field: 'slo.denominator'
    }
  }
};
function buildWindowAgg(id, type, threshold, slo, dateRange) {
  const aggs = _sloSchema.timeslicesBudgetingMethodSchema.is(slo.budgetingMethod) ? TIMESLICE_AGGS : OCCURRENCE_AGGS;
  return {
    [`${id}_${type}`]: {
      filter: {
        range: {
          '@timestamp': {
            gte: dateRange.from.toISOString(),
            lt: dateRange.to.toISOString()
          }
        }
      },
      aggs
    },
    [generateBurnRateKey(id, type)]: {
      bucket_script: {
        buckets_path: {
          good: `${id}_${type}>good`,
          total: `${id}_${type}>total`
        },
        script: {
          source: 'params.total != null && params.total > 0 ? (1 - (params.good / params.total)) / (1 - params.target) : 0',
          params: {
            target: slo.objective.target
          }
        }
      }
    },
    [generateAboveThresholdKey(id, type)]: {
      bucket_script: {
        buckets_path: {
          burnRate: generateBurnRateKey(id, type)
        },
        script: {
          source: 'params.burnRate >= params.threshold ? 1 : 0',
          params: {
            threshold
          }
        }
      }
    }
  };
}
function buildWindowAggs(startedAt, slo, burnRateWindows, delayInSeconds = 0) {
  return burnRateWindows.reduce((acc, winDef, index) => {
    const shortDateRange = (0, _get_lookback_date_range.getLookbackDateRange)(startedAt, winDef.shortDuration, delayInSeconds);
    const longDateRange = (0, _get_lookback_date_range.getLookbackDateRange)(startedAt, winDef.longDuration, delayInSeconds);
    const windowId = generateWindowId(index);
    return {
      ...acc,
      ...buildWindowAgg(windowId, SHORT_WINDOW, winDef.burnRateThreshold, slo, shortDateRange),
      ...buildWindowAgg(windowId, LONG_WINDOW, winDef.burnRateThreshold, slo, longDateRange)
    };
  }, {});
}
function buildEvaluation(burnRateWindows) {
  const bucketsPath = burnRateWindows.reduce((acc, _winDef, index) => {
    const windowId = `${WINDOW}_${index}`;
    return {
      ...acc,
      [generateAboveThresholdKey(windowId, SHORT_WINDOW)]: generateAboveThresholdKey(windowId, SHORT_WINDOW),
      [generateAboveThresholdKey(windowId, LONG_WINDOW)]: generateAboveThresholdKey(windowId, LONG_WINDOW)
    };
  }, {});
  const source = burnRateWindows.reduce((acc, _windDef, index) => {
    const windowId = `${WINDOW}_${index}`;
    const OP = acc ? ' || ' : '';
    return `${acc}${OP}(params.${generateAboveThresholdKey(windowId, SHORT_WINDOW)} == 1 && params.${generateAboveThresholdKey(windowId, LONG_WINDOW)} == 1)`;
  }, '');
  return {
    evaluation: {
      bucket_selector: {
        buckets_path: bucketsPath,
        script: {
          source
        }
      }
    }
  };
}
function buildQuery(startedAt, slo, params, afterKey) {
  const delayInSeconds = (0, _get_delay_in_seconds_from_slo.getDelayInSecondsFromSLO)(slo);
  const burnRateWindows = params.windows.map(winDef => {
    return {
      ...winDef,
      longDuration: new _models.Duration(winDef.longWindow.value, (0, _models.toDurationUnit)(winDef.longWindow.unit)),
      shortDuration: new _models.Duration(winDef.shortWindow.value, (0, _models.toDurationUnit)(winDef.shortWindow.unit))
    };
  });
  const longestLookbackWindow = burnRateWindows.reduce((acc, winDef) => {
    return winDef.longDuration.isShorterThan(acc.longDuration) ? acc : winDef;
  }, burnRateWindows[0]);
  const longestDateRange = (0, _get_lookback_date_range.getLookbackDateRange)(startedAt, longestLookbackWindow.longDuration, delayInSeconds);
  return {
    size: 0,
    query: {
      bool: {
        filter: [{
          term: {
            'slo.id': slo.id
          }
        }, {
          term: {
            'slo.revision': slo.revision
          }
        }, {
          range: {
            '@timestamp': {
              gte: longestDateRange.from.toISOString(),
              lt: longestDateRange.to.toISOString()
            }
          }
        }]
      }
    },
    aggs: {
      instances: {
        composite: {
          ...(afterKey ? {
            after: afterKey
          } : {}),
          size: 1000,
          sources: [{
            instanceId: {
              terms: {
                field: 'slo.instanceId'
              }
            }
          }]
        },
        aggs: {
          ...buildWindowAggs(startedAt, slo, burnRateWindows, delayInSeconds),
          ...buildEvaluation(burnRateWindows)
        }
      }
    }
  };
}