"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AutocompleteListener = void 0;
exports.nonNullable = nonNullable;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _dymanic_item = require("./dymanic_item");
var _esql_parser = require("../../antlr/esql_parser");
var _autocomplete_definitions = require("./autocomplete_definitions");
var _operators_commands = require("./autocomplete_definitions/operators_commands");
var _date_math_expressions = require("./autocomplete_definitions/date_math_expressions");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function nonNullable(v) {
  return v != null;
}
class AutocompleteListener {
  constructor() {
    (0, _defineProperty2.default)(this, "suggestions", []);
    (0, _defineProperty2.default)(this, "userDefinedVariables", {
      sourceIdentifiers: [],
      policyIdentifiers: []
    });
    (0, _defineProperty2.default)(this, "tables", []);
    (0, _defineProperty2.default)(this, "parentContext", void 0);
  }
  get fields() {
    return [_dymanic_item.DynamicAutocompleteItem.FieldIdentifier];
  }
  get policies() {
    return [_dymanic_item.DynamicAutocompleteItem.PolicyIdentifier];
  }
  get policyFields() {
    return [_dymanic_item.DynamicAutocompleteItem.PolicyFieldIdentifier];
  }
  get policyMatchingField() {
    return [_dymanic_item.DynamicAutocompleteItem.PolicyMatchingFieldIdentifier];
  }
  get hasSuggestions() {
    return Boolean(this.suggestions.length);
  }
  isTerminalNodeExists(node) {
    var _node$payload;
    return node && ((_node$payload = node.payload) === null || _node$payload === void 0 ? void 0 : _node$payload.startIndex) >= 0;
  }
  inspectOperatorExpressionContext(context, innerScope) {
    var _context$operatorExpr;
    if (!context) {
      return false;
    }
    if (Array.isArray(context)) {
      return context.some(c => this.inspectOperatorExpressionContext(c, innerScope));
    }
    if ((_context$operatorExpr = context.operatorExpression()) !== null && _context$operatorExpr !== void 0 && _context$operatorExpr.length) {
      return this.inspectOperatorExpressionContext(context.operatorExpression(), innerScope);
    }
    if (context.primaryExpression()) {
      var _context$primaryExpre;
      return Boolean((_context$primaryExpre = context.primaryExpression()) === null || _context$primaryExpre === void 0 ? void 0 : _context$primaryExpre[innerScope]());
    }
    return false;
  }
  hasDateExpressionTerminalNode(context) {
    return this.inspectOperatorExpressionContext(context, 'dateExpression');
  }
  hasOnlyConstantDefined(context) {
    return this.inspectOperatorExpressionContext(context, 'constant');
  }
  applyConditionalSuggestion(skipDefinitions, targetDefinition, context) {
    if (!skipDefinitions.find(i => i === targetDefinition) && this.parentContext === context) {
      return targetDefinition;
    }
  }
  getEndCommandSuggestions(skipDefinitions = []) {
    return [_autocomplete_definitions.pipeDefinition, this.applyConditionalSuggestion(skipDefinitions, _autocomplete_definitions.byOperatorDefinition, _esql_parser.esql_parser.STATS), this.applyConditionalSuggestion(skipDefinitions, _operators_commands.onOperatorDefinition, _esql_parser.esql_parser.ENRICH), this.applyConditionalSuggestion(skipDefinitions, _operators_commands.withOperatorDefinition, _esql_parser.esql_parser.ENRICH)].filter(nonNullable);
  }
  getNewVarName() {
    const vars = this.tables.flat();
    let index = 0;
    while (true) {
      const value = `var${index}`;
      if (!vars.includes(value)) {
        return value;
      }
      index++;
    }
  }
  getAutocompleteSuggestions() {
    return {
      suggestions: this.suggestions,
      userDefinedVariables: this.userDefinedVariables
    };
  }

  /** ESQLParserListener fields  **/

  enterSourceCommand(ctx) {
    this.suggestions = [];
  }
  exitSourceCommand(ctx) {
    if (ctx.exception) {
      this.suggestions = _autocomplete_definitions.sourceCommandsDefinitions;
    }
  }
  enterSourceIdentifier(ctx) {
    this.suggestions = [_dymanic_item.DynamicAutocompleteItem.SourceIdentifier];
  }
  exitSourceIdentifier(ctx) {
    if (!ctx.childCount) {
      this.suggestions = [_dymanic_item.DynamicAutocompleteItem.SourceIdentifier];
    } else if (!ctx.exception && ctx.text) {
      this.userDefinedVariables.sourceIdentifiers.push(ctx.text);
    }
  }
  enterProcessingCommand(ctx) {
    this.tables.push([]);
    this.suggestions = [];
    this.parentContext = undefined;
  }
  exitProcessingCommand(ctx) {
    if (ctx.exception) {
      this.suggestions = _autocomplete_definitions.processingCommandsDefinitions;
    }
    this.parentContext = undefined;
  }
  enterStatsCommand(ctx) {
    this.suggestions = [];
    this.parentContext = _esql_parser.esql_parser.STATS;
    const fn = ctx.fields();
    if (!fn) {
      this.suggestions = [(0, _autocomplete_definitions.buildNewVarDefinition)(this.getNewVarName())];
      return;
    }
  }
  enterEvalCommand(ctx) {
    this.suggestions = [];
    this.parentContext = _esql_parser.esql_parser.EVAL;
  }
  exitStatsCommand(ctx) {
    const qn = ctx.qualifiedNames();
    if (qn && qn.text) {
      this.suggestions = this.getEndCommandSuggestions([_autocomplete_definitions.byOperatorDefinition]);
    }
  }
  exitKeepCommand(ctx) {
    const qn = ctx.qualifiedNames();
    if (qn && qn.text) {
      if (qn.text.slice(-1) !== ',') {
        this.suggestions = this.getEndCommandSuggestions();
      }
    }
  }
  exitDropCommand(ctx) {
    const qn = ctx.qualifiedNames();
    if (qn && qn.text) {
      if (qn.text.slice(-1) !== ',') {
        this.suggestions = this.getEndCommandSuggestions();
      }
    }
  }
  enterRenameCommand(ctx) {
    this.parentContext = _esql_parser.esql_parser.RENAME;
  }
  exitRenameCommand(ctx) {
    const rc = ctx.renameClause();
    const commaExists = ctx.COMMA();
    if (!rc[0].exception) {
      const qn = rc[0].renameVariable();
      const asExists = this.isTerminalNodeExists(rc[0].AS());
      if (asExists && qn && !qn.text) {
        this.suggestions = [];
      }
      if (qn && qn.text) {
        if (!commaExists.length) {
          this.suggestions = this.getEndCommandSuggestions();
        }
      }
    }
  }
  exitDissectCommand(ctx) {
    const qn = ctx.qualifiedNames();
    const pattern = ctx.string();
    if (qn && qn.text && pattern && pattern.text && pattern.text !== '<missing STRING>') {
      this.suggestions = this.getEndCommandSuggestions();
    }
  }
  exitGrokCommand(ctx) {
    const qn = ctx.qualifiedNames();
    const pattern = ctx.string();
    if (qn && qn.text && pattern && pattern.text && pattern.text !== '<missing STRING>') {
      this.suggestions = this.getEndCommandSuggestions();
    }
  }
  exitMvExpandCommand(ctx) {
    const qn = ctx.qualifiedNames();
    if (qn && qn.text) {
      this.suggestions = this.getEndCommandSuggestions();
    }
  }
  exitQualifiedName(ctx) {
    const isInEval = this.parentContext === _esql_parser.esql_parser.EVAL;
    const isInStats = this.parentContext === _esql_parser.esql_parser.STATS;
    const isInRename = this.parentContext === _esql_parser.esql_parser.RENAME;
    if (this.parentContext && isInRename) {
      if (!ctx.exception && ctx.text) {
        this.suggestions = [_autocomplete_definitions.asOperatorDefinition];
      }
    }
    if (this.parentContext && (isInStats || isInEval)) {
      this.suggestions = [...this.getEndCommandSuggestions(), ...(isInEval ? _autocomplete_definitions.mathOperatorsCommandsDefinitions : [])];
    }
    if (ctx.identifier().some(i => !(this.isTerminalNodeExists(i.QUOTED_IDENTIFIER()) || this.isTerminalNodeExists(i.UNQUOTED_IDENTIFIER())))) {
      this.suggestions = this.fields;
    }
  }
  enterField(ctx) {
    this.suggestions = [];
  }
  exitField(ctx) {
    const hasAssign = this.isTerminalNodeExists(ctx.ASSIGN());
    if (ctx.exception) {
      if (!hasAssign) {
        this.suggestions = [(0, _autocomplete_definitions.buildNewVarDefinition)(this.getNewVarName())];
        return;
      }
    } else {
      if (!hasAssign) {
        this.suggestions = [_autocomplete_definitions.assignOperatorDefinition];
      }
    }
  }
  exitUserVariable(ctx) {
    if (!ctx.exception && ctx.text) {
      var _this$tables$at;
      (_this$tables$at = this.tables.at(-1)) === null || _this$tables$at === void 0 ? void 0 : _this$tables$at.push(ctx.text);
    }
  }
  enterBooleanExpression(ctx) {
    this.suggestions = [];
  }
  exitBooleanExpression(ctx) {
    if (ctx.exception) {
      const ve = ctx.valueExpression();
      if (!ve) {
        if (this.parentContext === _esql_parser.esql_parser.STATS) {
          this.suggestions = [..._autocomplete_definitions.aggregationFunctionsDefinitions];
          return;
        }
        if (this.parentContext === _esql_parser.esql_parser.EVAL) {
          this.suggestions = [..._autocomplete_definitions.mathCommandDefinition];
          return;
        }
      }
    }
  }
  exitValueExpression(ctx) {
    const isInStats = this.parentContext === _esql_parser.esql_parser.STATS;
    const isInEval = this.parentContext === _esql_parser.esql_parser.EVAL;
    if (isInStats || isInEval) {
      const hasFN = ctx.tryGetToken(_esql_parser.esql_parser.UNARY_FUNCTION, 0) || ctx.tryGetToken(_esql_parser.esql_parser.MATH_FUNCTION, 0);
      const hasLP = ctx.tryGetToken(_esql_parser.esql_parser.LP, 0);
      const hasRP = ctx.tryGetToken(_esql_parser.esql_parser.RP, 0);
      // TODO: handle also other math signs later on
      const hasPlusOrMinus = ctx.tryGetToken(_esql_parser.esql_parser.PLUS, 0) || ctx.tryGetToken(_esql_parser.esql_parser.MINUS, 0);
      const hasDateLiteral = ctx.tryGetToken(_esql_parser.esql_parser.DATE_LITERAL, 0);
      const isInDurationMode = hasDateLiteral || hasFN && (0, _helpers.isDateFunction)(hasFN.text);
      if (hasPlusOrMinus && this.isTerminalNodeExists(hasPlusOrMinus)) {
        if (isInEval) {
          this.suggestions = isInDurationMode ?
          // eval a = 1 year + ||  eval a = date_trunc(1 year, date) -
          [..._autocomplete_definitions.mathCommandDefinition.filter(({
            label
          }) => (0, _helpers.isDateFunction)(String(label))), ...(0, _helpers.getDurationItemsWithQuantifier)()] :
          // eval a = 1 + || eval a = abs(b) -
          [...this.fields, ..._autocomplete_definitions.mathCommandDefinition];
        } else {
          this.suggestions = [...this.fields, ..._autocomplete_definitions.aggregationFunctionsDefinitions];
        }
        return;
      }

      // Monaco will auto close the brackets but the language listener will not pick up yet this auto-change.
      // We try to inject it outside but it won't cover all scenarios
      if (hasFN) {
        if (!hasLP) {
          this.suggestions = [_autocomplete_definitions.openBracketDefinition];
          return;
        }
        this.suggestions = [];
        if (!hasRP) {
          if (ctx.childCount === 3) {
            // TODO: improve here to suggest comma if signature has multiple args
            this.suggestions.push(_autocomplete_definitions.closeBracketDefinition);
          }
        }
        this.suggestions.push(...this.fields);
        // Need to get the function name from the previous node (current is "(" )
        const fnName = hasFN.text;
        const fnsToCheck = isInEval ? _autocomplete_definitions.mathCommandDefinition : _autocomplete_definitions.aggregationFunctionsDefinitions;
        if (fnName && fnsToCheck.some(({
          label
        }) => label === fnName)) {
          // push date suggestions only for date functions
          // TODO: improve this checks
          if (isInEval && (0, _helpers.isDateFunction)(fnName)) {
            if (!ctx.tryGetToken(_esql_parser.esql_parser.DATE_LITERAL, 0)) {
              this.suggestions.push(
              // if it's just after the open bracket, suggest also a number together with a date period,
              // otherwise just the date period unit
              ...((0, _helpers.endsWithOpenBracket)(ctx.text) ? (0, _helpers.getDurationItemsWithQuantifier)() : _date_math_expressions.dateExpressionDefinitions));
            }
          }
        }
        return;
      } else {
        if (ctx.childCount === 1) {
          if (ctx.text && ctx.text.indexOf('(') === -1) {
            this.suggestions = [..._autocomplete_definitions.mathOperatorsCommandsDefinitions];
            if (isInEval) {
              // eval a = 1 || eval a = 1 year + 1
              if (this.hasDateExpressionTerminalNode(ctx.operatorExpression()) || this.hasOnlyConstantDefined(ctx.operatorExpression())) {
                this.suggestions = [...(0, _helpers.getDateMathOperation)(), ..._date_math_expressions.dateExpressionDefinitions];
              }
            }
            if (isInStats) {
              this.suggestions.push(..._autocomplete_definitions.aggregationFunctionsDefinitions);
            }
            this.suggestions.push(...this.getEndCommandSuggestions());
          }
          return;
        }
      }
      this.suggestions = [...this.fields];
      if (ctx.exception && isInEval) {
        // case: eval a = x * or / <here>
        this.suggestions.push(..._autocomplete_definitions.mathCommandDefinition);
      }
      this.suggestions.push(...this.getEndCommandSuggestions());
    }
  }
  enterWhereBooleanExpression(ctx) {
    this.suggestions = [];
  }
  enterWhereCommand(ctx) {
    this.suggestions = [];
    this.parentContext = _esql_parser.esql_parser.WHERE;
  }
  enterEnrichCommand(ctx) {
    this.suggestions = [];
    this.parentContext = _esql_parser.esql_parser.ENRICH;
  }
  exitEnrichCommand(ctx) {
    const policyName = ctx.enrichIdentifier().text;
    if (policyName && !this.userDefinedVariables.policyIdentifiers.includes(policyName)) {
      this.userDefinedVariables.policyIdentifiers.push(policyName);
    }
    if (this.parentContext === _esql_parser.esql_parser.WITH) {
      return;
    }
    if (!policyName) {
      this.suggestions = this.policies;
    }
    if (policyName) if (this.parentContext === _esql_parser.esql_parser.ENRICH) {
      const hasOn = this.isTerminalNodeExists(ctx.ON());
      if (hasOn && !ctx._matchField.text) {
        this.suggestions = this.policyMatchingField;
      } else {
        this.suggestions = this.getEndCommandSuggestions(hasOn ? [_operators_commands.onOperatorDefinition] : undefined);
      }
    }
  }
  enterEnrichWithClause(ctx) {
    this.suggestions = [];
    this.parentContext = _esql_parser.esql_parser.WITH;
  }
  exitEnrichWithClause(ctx) {
    var _ctx$_newName;
    const hasAssign = this.isTerminalNodeExists(ctx.ASSIGN());
    // Note: this gets filled only after the assign operation :(
    if ((_ctx$_newName = ctx._newName) !== null && _ctx$_newName !== void 0 && _ctx$_newName.text) {
      var _this$tables$at2;
      (_this$tables$at2 = this.tables.at(-1)) === null || _this$tables$at2 === void 0 ? void 0 : _this$tables$at2.push(ctx._newName.text);
    }
    if (!ctx.exception && ctx.enrichFieldIdentifier().length === 1) {
      // if it's after the assign operator, then suggest the fields from the policy
      // TODO: need to check if the enrichFieldIdentifier given is a policyField or not and decide whether append the assignOperator
      this.suggestions = !hasAssign ? [_autocomplete_definitions.assignOperatorDefinition, ...this.getEndCommandSuggestions()] : this.policyFields;
    } else {
      var _ctx$_enrichField;
      this.suggestions = [];
      if (!hasAssign) {
        this.suggestions.push((0, _autocomplete_definitions.buildNewVarDefinition)(this.getNewVarName()));
      }
      if (!((_ctx$_enrichField = ctx._enrichField) !== null && _ctx$_enrichField !== void 0 && _ctx$_enrichField.text)) {
        this.suggestions.push(...this.policyFields);
      }
      if (this.suggestions.length === 0) {
        this.suggestions = this.getEndCommandSuggestions([_operators_commands.onOperatorDefinition, _operators_commands.withOperatorDefinition]);
      }
    }
  }
  exitWhereCommand(ctx) {
    const booleanExpression = ctx.whereBooleanExpression();
    if (booleanExpression.exception) {
      if (!booleanExpression.text) {
        this.suggestions = [..._autocomplete_definitions.whereCommandDefinition, ...this.fields];
        return;
      }
      this.suggestions = this.fields;
      return;
    } else {
      const innerBooleanExpressions = booleanExpression.getRuleContexts(_esql_parser.WhereBooleanExpressionContext);
      const regexBooleanExpression = booleanExpression.getRuleContexts(_esql_parser.RegexBooleanExpressionContext);
      if (booleanExpression.WHERE_FUNCTIONS()) {
        if (booleanExpression.COMMA().length) {
          this.suggestions = [];
          return;
        }
      }
      if (regexBooleanExpression.length) {
        this.suggestions = [];
        return;
      }
      if (innerBooleanExpressions.some(be => be.exception)) {
        this.suggestions = this.fields;
        return;
      }
    }
    if (!this.hasSuggestions && !booleanExpression.WHERE_FUNCTIONS()) {
      this.suggestions = _autocomplete_definitions.comparisonCommandsDefinitions;
    }
  }
  exitComparison(ctx) {
    const operatorExpression = ctx.operatorExpression();
    if (operatorExpression.some(o => o.exception)) {
      this.suggestions = this.fields;
      return;
    }
    this.suggestions = [..._autocomplete_definitions.comparisonOperatorsCommandsDefinitions, ...this.getEndCommandSuggestions()];
  }
  exitOrderExpression(ctx) {
    if (ctx.booleanExpression().exception) {
      this.suggestions = this.fields;
      return;
    }
    if (!this.isTerminalNodeExists(ctx.ORDERING())) {
      this.suggestions = _autocomplete_definitions.orderingCommandsDefinitions;
      return;
    }
    if (!this.isTerminalNodeExists(ctx.NULLS_ORDERING())) {
      this.suggestions = [_autocomplete_definitions.nullsCommandsDefinition];
      return;
    }
    if (!this.isTerminalNodeExists(ctx.NULLS_ORDERING_DIRECTION())) {
      this.suggestions = _autocomplete_definitions.nullsOrderingCommandsDefinitions;
      return;
    }
  }
  exitLimitCommand(ctx) {
    const DEFAULT_LIMIT_SIZE = 1000;
    if (!this.isTerminalNodeExists(ctx.INTEGER_LITERAL())) {
      this.suggestions = (0, _autocomplete_definitions.buildConstantsDefinitions)([DEFAULT_LIMIT_SIZE.toString()], '');
    } else {
      this.suggestions = this.getEndCommandSuggestions();
    }
  }
}
exports.AutocompleteListener = AutocompleteListener;