"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.syncMvtSourceData = syncMvtSourceData;
var _uuid = require("uuid");
var _constants = require("../../../../../common/constants");
var _can_skip_fetch = require("../../../util/can_skip_fetch");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// shape of sourceDataRequest.getData()

async function syncMvtSourceData({
  buffer,
  hasLabels,
  layerId,
  layerName,
  prevDataRequest,
  requestMeta,
  source,
  syncContext
}) {
  const requestToken = Symbol(`${layerId}-${_constants.SOURCE_DATA_REQUEST_ID}`);
  const prevData = prevDataRequest ? prevDataRequest.getData() : undefined;
  if (prevData) {
    const noChangesInSourceState = prevData.tileSourceLayer === source.getTileSourceLayer() && prevData.tileMinZoom === source.getMinZoom() && prevData.tileMaxZoom === source.getMaxZoom();
    const noChangesInSearchState = await (0, _can_skip_fetch.canSkipSourceUpdate)({
      extentAware: false,
      // spatial extent knowledge is already fully automated by tile-loading based on pan-zooming
      source,
      prevDataRequest,
      nextRequestMeta: requestMeta,
      getUpdateDueToTimeslice: timeslice => {
        return true;
      }
    });
    const canSkip = !syncContext.forceRefreshDueToDrawing && noChangesInSourceState && noChangesInSearchState && prevData.hasLabels === hasLabels && prevData.buffer === buffer;
    if (canSkip) {
      return;
    }
  }
  syncContext.startLoading(_constants.SOURCE_DATA_REQUEST_ID, requestToken, requestMeta);
  try {
    const refreshToken = !prevData || syncContext.forceRefreshDueToDrawing || requestMeta.isForceRefresh && requestMeta.applyForceRefresh ? (0, _uuid.v4)() : prevData.refreshToken;
    const tileUrl = await source.getTileUrl(requestMeta, refreshToken, hasLabels, buffer);
    if (source.isESSource()) {
      syncContext.inspectorAdapters.vectorTiles.addLayer(layerId, layerName, tileUrl);
    }
    const sourceData = {
      tileUrl,
      tileSourceLayer: source.getTileSourceLayer(),
      tileMinZoom: source.getMinZoom(),
      tileMaxZoom: source.getMaxZoom(),
      refreshToken,
      hasLabels,
      buffer
    };
    syncContext.stopLoading(_constants.SOURCE_DATA_REQUEST_ID, requestToken, sourceData, {});
  } catch (error) {
    syncContext.onLoadError(_constants.SOURCE_DATA_REQUEST_ID, requestToken, error);
  }
}