"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.overallSumOperation = exports.overallMinOperation = exports.overallMaxOperation = exports.overallAverageOperation = void 0;
var _i18n = require("@kbn/i18n");
var _utils = require("./utils");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function buildOverallMetricOperation({
  type,
  displayName,
  ofName,
  description,
  metric
}) {
  return {
    type,
    priority: 1,
    displayName,
    input: 'fullReference',
    selectionStyle: 'hidden',
    requiredReferences: [{
      input: ['field', 'managedReference', 'fullReference'],
      validateMetadata: meta => meta.dataType === 'number' && !meta.isBucketed
    }],
    getPossibleOperation: () => {
      return {
        dataType: 'number',
        isBucketed: false,
        scale: 'ratio'
      };
    },
    getDefaultLabel: (column, columns, indexPattern) => {
      var _indexPattern$getFiel;
      const ref = columns[column.references[0]];
      return ofName(ref && 'sourceField' in ref ? indexPattern === null || indexPattern === void 0 ? void 0 : (_indexPattern$getFiel = indexPattern.getFieldByName(ref.sourceField)) === null || _indexPattern$getFiel === void 0 ? void 0 : _indexPattern$getFiel.displayName : undefined);
    },
    toExpression: (layer, columnId) => {
      return (0, _utils.optionallHistogramBasedOperationToExpression)(layer, columnId, 'overall_metric', {
        metric: [metric]
      });
    },
    buildColumn: ({
      referenceIds,
      previousColumn,
      layer,
      indexPattern
    }, columnParams) => {
      var _indexPattern$getFiel2;
      const ref = layer.columns[referenceIds[0]];
      return {
        label: ofName(ref && 'sourceField' in ref ? (_indexPattern$getFiel2 = indexPattern.getFieldByName(ref.sourceField)) === null || _indexPattern$getFiel2 === void 0 ? void 0 : _indexPattern$getFiel2.displayName : undefined),
        dataType: 'number',
        operationType: `overall_${metric}`,
        isBucketed: false,
        scale: 'ratio',
        references: referenceIds,
        params: (0, _helpers.getFormatFromPreviousColumn)(previousColumn)
      };
    },
    isTransferable: () => {
      return true;
    },
    filterable: false,
    shiftable: false,
    documentation: {
      section: 'calculation',
      signature: _i18n.i18n.translate('xpack.lens.indexPattern.overall_metric', {
        defaultMessage: 'metric: number'
      }),
      description
    }
  };
}
const overallSumOperation = buildOverallMetricOperation({
  type: 'overall_sum',
  displayName: _i18n.i18n.translate('xpack.lens.indexPattern.overallSum', {
    defaultMessage: 'Overall sum'
  }),
  ofName: name => {
    return _i18n.i18n.translate('xpack.lens.indexPattern.overallSumOf', {
      defaultMessage: 'Overall sum of {name}',
      values: {
        name: name !== null && name !== void 0 ? name : _i18n.i18n.translate('xpack.lens.indexPattern.incompleteOperation', {
          defaultMessage: '(incomplete)'
        })
      }
    });
  },
  metric: 'sum',
  description: _i18n.i18n.translate('xpack.lens.indexPattern.overall_sum.documentation.markdown', {
    defaultMessage: `
Calculates the sum of a metric of all data points of a series in the current chart. A series is defined by a dimension using a date histogram or interval function.
Other dimensions breaking down the data like top values or filter are treated as separate series.

If no date histograms or interval functions are used in the current chart, \`overall_sum\` is calculating the sum over all dimensions no matter the used function.

Example: Percentage of total
\`sum(bytes) / overall_sum(sum(bytes))\`
      `
  })
});
exports.overallSumOperation = overallSumOperation;
const overallMinOperation = buildOverallMetricOperation({
  type: 'overall_min',
  displayName: _i18n.i18n.translate('xpack.lens.indexPattern.overallMin', {
    defaultMessage: 'Overall min'
  }),
  ofName: name => {
    return _i18n.i18n.translate('xpack.lens.indexPattern.overallMinOf', {
      defaultMessage: 'Overall min of {name}',
      values: {
        name: name !== null && name !== void 0 ? name : _i18n.i18n.translate('xpack.lens.indexPattern.incompleteOperation', {
          defaultMessage: '(incomplete)'
        })
      }
    });
  },
  metric: 'min',
  description: _i18n.i18n.translate('xpack.lens.indexPattern.overall_min.documentation.markdown', {
    defaultMessage: `
Calculates the minimum of a metric for all data points of a series in the current chart. A series is defined by a dimension using a date histogram or interval function.
Other dimensions breaking down the data like top values or filter are treated as separate series.

If no date histograms or interval functions are used in the current chart, \`overall_min\` is calculating the minimum over all dimensions no matter the used function

Example: Percentage of range
\`(sum(bytes) - overall_min(sum(bytes)) / (overall_max(sum(bytes)) - overall_min(sum(bytes)))\`
      `
  })
});
exports.overallMinOperation = overallMinOperation;
const overallMaxOperation = buildOverallMetricOperation({
  type: 'overall_max',
  displayName: _i18n.i18n.translate('xpack.lens.indexPattern.overallMax', {
    defaultMessage: 'Overall max'
  }),
  ofName: name => {
    return _i18n.i18n.translate('xpack.lens.indexPattern.overallMaxOf', {
      defaultMessage: 'Overall max of {name}',
      values: {
        name: name !== null && name !== void 0 ? name : _i18n.i18n.translate('xpack.lens.indexPattern.incompleteOperation', {
          defaultMessage: '(incomplete)'
        })
      }
    });
  },
  metric: 'max',
  description: _i18n.i18n.translate('xpack.lens.indexPattern.overall_max.documentation.markdown', {
    defaultMessage: `
Calculates the maximum of a metric for all data points of a series in the current chart. A series is defined by a dimension using a date histogram or interval function.
Other dimensions breaking down the data like top values or filter are treated as separate series.

If no date histograms or interval functions are used in the current chart, \`overall_max\` is calculating the maximum over all dimensions no matter the used function

Example: Percentage of range
\`(sum(bytes) - overall_min(sum(bytes))) / (overall_max(sum(bytes)) - overall_min(sum(bytes)))\`
      `
  })
});
exports.overallMaxOperation = overallMaxOperation;
const overallAverageOperation = buildOverallMetricOperation({
  type: 'overall_average',
  displayName: _i18n.i18n.translate('xpack.lens.indexPattern.overallMax', {
    defaultMessage: 'Overall max'
  }),
  ofName: name => {
    return _i18n.i18n.translate('xpack.lens.indexPattern.overallAverageOf', {
      defaultMessage: 'Overall average of {name}',
      values: {
        name: name !== null && name !== void 0 ? name : _i18n.i18n.translate('xpack.lens.indexPattern.incompleteOperation', {
          defaultMessage: '(incomplete)'
        })
      }
    });
  },
  metric: 'average',
  description: _i18n.i18n.translate('xpack.lens.indexPattern.overall_average.documentation.markdown', {
    defaultMessage: `
Calculates the average of a metric for all data points of a series in the current chart. A series is defined by a dimension using a date histogram or interval function.
Other dimensions breaking down the data like top values or filter are treated as separate series.

If no date histograms or interval functions are used in the current chart, \`overall_average\` is calculating the average over all dimensions no matter the used function

Example: Divergence from the mean:
\`sum(bytes) - overall_average(sum(bytes))\`
      `
  })
});
exports.overallAverageOperation = overallAverageOperation;