"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSuggestions = exports.getQueryColumns = exports.getLensAttributes = void 0;
var _i18n = require("@kbn/i18n");
var _esQuery = require("@kbn/es-query");
var _fetch_data_from_aggregate_query = require("../../../datasources/text_based/fetch_data_from_aggregate_query");
var _lens_suggestions_api = require("../../../lens_suggestions_api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getQueryColumns = async (query, dataView, deps) => {
  // Fetching only columns for ES|QL for performance reasons with limit 0
  // Important note: ES doesnt return the warnings for 0 limit,
  // I am skipping them in favor of performance now
  // but we should think another way to get them (from Lens embeddable or store)
  const performantQuery = {
    ...query
  };
  if ('esql' in performantQuery && performantQuery.esql) {
    performantQuery.esql = `${performantQuery.esql} | limit 0`;
  }
  const table = await (0, _fetch_data_from_aggregate_query.fetchDataFromAggregateQuery)(performantQuery, dataView, deps.data, deps.expressions);
  return table === null || table === void 0 ? void 0 : table.columns;
};
exports.getQueryColumns = getQueryColumns;
const getSuggestions = async (query, deps, datasourceMap, visualizationMap, adHocDataViews, setErrors) => {
  try {
    var _dataView$fields$getB, _suggestionsApi;
    let indexPattern = '';
    if ('sql' in query) {
      indexPattern = (0, _esQuery.getIndexPatternFromSQLQuery)(query.sql);
    }
    if ('esql' in query) {
      indexPattern = (0, _esQuery.getIndexPatternFromESQLQuery)(query.esql);
    }
    const dataViewSpec = adHocDataViews.find(adHoc => {
      return adHoc.name === indexPattern;
    });
    const dataView = await deps.dataViews.create(dataViewSpec !== null && dataViewSpec !== void 0 ? dataViewSpec : {
      title: indexPattern
    });
    if (((_dataView$fields$getB = dataView.fields.getByName('@timestamp')) === null || _dataView$fields$getB === void 0 ? void 0 : _dataView$fields$getB.type) === 'date' && !dataViewSpec) {
      dataView.timeFieldName = '@timestamp';
    }
    const columns = await getQueryColumns(query, dataView, deps);
    const context = {
      dataViewSpec: dataView === null || dataView === void 0 ? void 0 : dataView.toSpec(),
      fieldName: '',
      textBasedColumns: columns,
      query
    };
    const allSuggestions = (_suggestionsApi = (0, _lens_suggestions_api.suggestionsApi)({
      context,
      dataView,
      datasourceMap,
      visualizationMap
    })) !== null && _suggestionsApi !== void 0 ? _suggestionsApi : [];

    // Lens might not return suggestions for some cases, i.e. in case of errors
    if (!allSuggestions.length) return undefined;
    const firstSuggestion = allSuggestions[0];
    const attrs = getLensAttributes({
      filters: [],
      query,
      suggestion: firstSuggestion,
      dataView
    });
    return attrs;
  } catch (e) {
    setErrors([e]);
  }
  return undefined;
};
exports.getSuggestions = getSuggestions;
const getLensAttributes = ({
  filters,
  query,
  suggestion,
  dataView
}) => {
  var _dataView$id;
  const suggestionDatasourceState = Object.assign({}, suggestion === null || suggestion === void 0 ? void 0 : suggestion.datasourceState);
  const suggestionVisualizationState = Object.assign({}, suggestion === null || suggestion === void 0 ? void 0 : suggestion.visualizationState);
  const datasourceStates = suggestion && suggestion.datasourceState ? {
    [suggestion.datasourceId]: {
      ...suggestionDatasourceState
    }
  } : {
    formBased: {}
  };
  const visualization = suggestionVisualizationState;
  const attributes = {
    title: suggestion ? suggestion.title : _i18n.i18n.translate('xpack.lens.config.suggestion.title', {
      defaultMessage: 'New suggestion'
    }),
    references: [{
      id: (_dataView$id = dataView === null || dataView === void 0 ? void 0 : dataView.id) !== null && _dataView$id !== void 0 ? _dataView$id : '',
      name: `textBasedLanguages-datasource-layer-suggestion`,
      type: 'index-pattern'
    }],
    state: {
      datasourceStates,
      filters,
      query,
      visualization,
      ...(dataView && dataView.id && !dataView.isPersisted() && {
        adHocDataViews: {
          [dataView.id]: dataView.toSpec(false)
        }
      })
    },
    visualizationType: suggestion ? suggestion.visualizationId : 'lnsXY'
  };
  return attributes;
};
exports.getLensAttributes = getLensAttributes;