"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PhraseFilterManager = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _esQuery = require("@kbn/es-query");
var _filter_manager = require("./filter_manager");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

class PhraseFilterManager extends _filter_manager.FilterManager {
  constructor(controlId, fieldName, indexPatternId, indexPatternsService, queryFilter) {
    super(controlId, fieldName, indexPatternId, indexPatternsService, queryFilter);
  }
  createFilter(phrases) {
    const indexPattern = this.getIndexPattern();
    let newFilter;
    const value = indexPattern.fields.getByName(this.fieldName);
    if (!value) {
      throw new Error(`Unable to find field with name: ${this.fieldName} on indexPattern`);
    }
    if (phrases.length === 1) {
      newFilter = (0, _esQuery.buildPhraseFilter)(value, phrases[0], indexPattern);
    } else {
      newFilter = (0, _esQuery.buildPhrasesFilter)(value, phrases, indexPattern);
    }
    newFilter.meta.key = this.fieldName;
    newFilter.meta.controlledBy = this.controlId;
    return newFilter;
  }
  getValueFromFilterBar() {
    const kbnFilters = this.findFilters();
    if (kbnFilters.length === 0) {
      return;
    }
    const values = kbnFilters.map(kbnFilter => {
      return this.getValueFromFilter(kbnFilter);
    }).filter(value => value != null);
    if (values.length === 0) {
      return;
    }
    return values.reduce((accumulator, currentValue) => {
      return accumulator.concat(currentValue);
    }, []);
  }

  /**
   * Extract filtering value from kibana filters
   *
   * @param  {PhraseFilter} kbnFilter
   * @return {Array.<string>} array of values pulled from filter
   */
  getValueFromFilter(kbnFilter) {
    // bool filter - multiple phrase filters
    if (_lodash.default.has(kbnFilter, 'query.bool.should')) {
      return _lodash.default.get(kbnFilter, 'query.bool.should').map(kbnQueryFilter => {
        return this.getValueFromFilter(kbnQueryFilter);
      }).filter(value => {
        if (value) {
          return true;
        }
        return false;
      });
    }

    // scripted field filter
    if (_lodash.default.has(kbnFilter, 'query.script')) {
      return _lodash.default.get(kbnFilter, 'query.script.script.params.value');
    }

    // single phrase filter
    if ((0, _esQuery.isPhraseFilter)(kbnFilter)) {
      if ((0, _esQuery.getPhraseFilterField)(kbnFilter) !== this.fieldName) {
        return;
      }
      return (0, _esQuery.getPhraseFilterValue)(kbnFilter);
    }

    // single phrase filter from bool filter
    if (_lodash.default.has(kbnFilter, ['match_phrase', this.fieldName])) {
      return _lodash.default.get(kbnFilter, ['match_phrase', this.fieldName]);
    }
  }
}
exports.PhraseFilterManager = PhraseFilterManager;