"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useRequestObservable = void 0;
var _react = require("react");
var _operators = require("rxjs/operators");
var _rxjs = require("rxjs");
var _use_loading_state = require("./use_loading_state");
var _use_date_picker = require("./use_date_picker");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useRequestObservable = () => {
  const {
    requestState$,
    isAutoRefreshRequestPending$
  } = (0, _use_loading_state.useLoadingStateContext)();
  const {
    autoRefreshConfig$
  } = (0, _use_date_picker.useDatePickerContext)();
  const request$ = (0, _react.useMemo)(() => new _rxjs.Subject(), []);
  (0, _react.useEffect)(() => {
    // Subscribe to updates in the request$
    const subscription = request$.pipe(
    // Combine latest values from request$, autoRefreshConfig$, and pendingRequests$
    (0, _rxjs.withLatestFrom)(isAutoRefreshRequestPending$, autoRefreshConfig$),
    // Filter out requests if there are pending requests or autoRefresh is paused
    // Prevents aborting running requests when autoRefresh is on
    (0, _operators.filter)(([, isAutoRefreshRequestPending, autoRefreshConfig]) => {
      return !isAutoRefreshRequestPending || (autoRefreshConfig === null || autoRefreshConfig === void 0 ? void 0 : autoRefreshConfig.isPaused) === true;
    }), (0, _operators.mergeMap)(([requestFn]) => {
      // If request function is not defined, return an observable that emits null
      if (!requestFn) {
        return (0, _rxjs.of)(null);
      }
      requestState$.next('running');
      return (0, _rxjs.from)(requestFn()).pipe((0, _operators.map)(response => {
        requestState$.next('done');
        return response;
      }), (0, _rxjs.catchError)(error => {
        requestState$.next('error');
        throw error;
      }));
    })).subscribe();
    return () => {
      requestState$.next('done');
      subscription.unsubscribe();
    };
  }, [autoRefreshConfig$, isAutoRefreshRequestPending$, request$, requestState$]);
  return {
    request$
  };
};
exports.useRequestObservable = useRequestObservable;