"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleTransformReauthorizeAndStart = handleTransformReauthorizeAndStart;
var _lodash = require("lodash");
var _install = require("../../packages/install");
var _common = require("../../../../../common");
var _retry = require("../retry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function reauthorizeAndStartTransform({
  esClient,
  logger,
  transformId,
  secondaryAuth,
  meta
}) {
  try {
    await (0, _retry.retryTransientEsErrors)(() => esClient.transform.updateTransform({
      transform_id: transformId,
      body: {
        _meta: meta
      }
    }, {
      ...(secondaryAuth ? secondaryAuth : {})
    }), {
      logger,
      additionalResponseStatuses: [400]
    });
    logger.debug(`Updated transform: ${transformId}`);
  } catch (err) {
    logger.error(`Failed to update transform: ${transformId} because ${err}`);
    return {
      transformId,
      success: false,
      error: err
    };
  }
  try {
    const startedTransform = await (0, _retry.retryTransientEsErrors)(() => esClient.transform.startTransform({
      transform_id: transformId
    }, {
      ignore: [409]
    }), {
      logger,
      additionalResponseStatuses: [400]
    });
    logger.debug(`Started transform: ${transformId}`);
    return {
      transformId,
      success: startedTransform.acknowledged,
      error: null
    };
  } catch (err) {
    logger.error(`Failed to start transform: ${transformId} because ${err}`);
    return {
      transformId,
      success: false,
      error: err
    };
  }
}
async function handleTransformReauthorizeAndStart({
  esClient,
  savedObjectsClient,
  logger,
  pkgName,
  pkgVersion,
  transforms,
  secondaryAuth,
  username
}) {
  var _so$attributes$instal;
  if (!secondaryAuth) {
    throw Error('A valid secondary authorization with sufficient `manage_transform` permission is needed to re-authorize and start transforms. ' + 'This could be because security is not enabled, or API key cannot be generated.');
  }
  const transformInfos = await Promise.all(transforms.map(({
    transformId
  }) => (0, _retry.retryTransientEsErrors)(() => esClient.transform.getTransform({
    transform_id: transformId
  }, {
    ...(secondaryAuth ? secondaryAuth : {})
  }), {
    logger,
    additionalResponseStatuses: [400]
  })));
  const transformsMetadata = transformInfos.flat().map(t => {
    var _t$transforms;
    const transform = (_t$transforms = t.transforms) === null || _t$transforms === void 0 ? void 0 : _t$transforms[0];
    return {
      ...transform._meta,
      transformId: transform === null || transform === void 0 ? void 0 : transform.id
    };
  }).filter(t => (t === null || t === void 0 ? void 0 : t.run_as_kibana_system) === false);
  const shouldInstallSequentially = (0, _lodash.uniqBy)(transformsMetadata, 'order').length === transforms.length;
  let authorizedTransforms = [];
  if (shouldInstallSequentially) {
    const sortedTransformsMetadata = (0, _lodash.sortBy)(transformsMetadata, [t => {
      var _t$package;
      return (_t$package = t.package) === null || _t$package === void 0 ? void 0 : _t$package.name;
    }, t => t.fleet_transform_version, t => t.order]);
    for (const {
      transformId,
      ...meta
    } of sortedTransformsMetadata) {
      const authorizedTransform = await reauthorizeAndStartTransform({
        esClient,
        logger,
        transformId,
        secondaryAuth,
        meta: {
          ...meta,
          last_authorized_by: username
        }
      });
      authorizedTransforms.push(authorizedTransform);
    }
  } else {
    // Else, create & start all the transforms at once for speed
    const transformsPromises = transformsMetadata.map(async ({
      transformId,
      ...meta
    }) => {
      return await reauthorizeAndStartTransform({
        esClient,
        logger,
        transformId,
        secondaryAuth,
        meta: {
          ...meta,
          last_authorized_by: username
        }
      });
    });
    authorizedTransforms = await Promise.all(transformsPromises).then(results => results.flat());
  }
  const so = await savedObjectsClient.get(_common.PACKAGES_SAVED_OBJECT_TYPE, pkgName);
  const esReferences = (_so$attributes$instal = so.attributes.installed_es) !== null && _so$attributes$instal !== void 0 ? _so$attributes$instal : [];
  const successfullyAuthorizedTransforms = authorizedTransforms.filter(t => t.success);
  const authorizedTransformsRefs = successfullyAuthorizedTransforms.map(t => ({
    type: _common.ElasticsearchAssetType.transform,
    id: t.transformId,
    version: pkgVersion
  }));
  await (0, _install.updateEsAssetReferences)(savedObjectsClient, pkgName, esReferences, {
    assetsToRemove: authorizedTransformsRefs,
    assetsToAdd: authorizedTransformsRefs
  });
  return authorizedTransforms;
}