"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getUninstallTokensMetadataHandler = exports.getUninstallTokenHandler = void 0;
var _services = require("../../services");
var _errors = require("../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const UNINSTALL_TOKEN_SERVICE_UNAVAILABLE_ERROR = {
  statusCode: 500,
  body: {
    message: 'Uninstall Token Service is unavailable.'
  }
};
const getUninstallTokensMetadataHandler = async (context, request, response) => {
  const uninstallTokenService = _services.appContextService.getUninstallTokenService();
  if (!uninstallTokenService) {
    return response.customError(UNINSTALL_TOKEN_SERVICE_UNAVAILABLE_ERROR);
  }
  try {
    const {
      page = 1,
      perPage = 20,
      policyId
    } = request.query;
    const body = await uninstallTokenService.getTokenMetadata(policyId === null || policyId === void 0 ? void 0 : policyId.trim(), page, perPage, 'policy-elastic-agent-on-cloud');
    return response.ok({
      body
    });
  } catch (error) {
    return (0, _errors.defaultFleetErrorHandler)({
      error,
      response
    });
  }
};
exports.getUninstallTokensMetadataHandler = getUninstallTokensMetadataHandler;
const getUninstallTokenHandler = async (context, request, response) => {
  const uninstallTokenService = _services.appContextService.getUninstallTokenService();
  if (!uninstallTokenService) {
    return response.customError(UNINSTALL_TOKEN_SERVICE_UNAVAILABLE_ERROR);
  }
  try {
    const {
      uninstallTokenId
    } = request.params;
    const token = await uninstallTokenService.getToken(uninstallTokenId);
    if (token === null) {
      return response.notFound({
        body: {
          message: `Uninstall Token not found with id ${uninstallTokenId}`
        }
      });
    }
    const body = {
      item: token
    };
    return response.ok({
      body
    });
  } catch (error) {
    return (0, _errors.defaultFleetErrorHandler)({
      error,
      response
    });
  }
};
exports.getUninstallTokenHandler = getUninstallTokenHandler;