"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerRoutes = exports.getCheckPermissionsHandler = exports.generateServiceTokenHandler = void 0;
var _constants = require("../../constants");
var _constants2 = require("../../../common/constants");
var _services = require("../../services");
var _errors = require("../../errors");
var _types = require("../../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getCheckPermissionsHandler = async (context, request, response) => {
  const missingSecurityBody = {
    success: false,
    error: 'MISSING_SECURITY'
  };
  if (!_services.appContextService.getSecurityLicense().isEnabled()) {
    return response.ok({
      body: missingSecurityBody
    });
  } else {
    const fleetContext = await context.fleet;
    if (!fleetContext.authz.fleet.all) {
      return response.ok({
        body: {
          success: false,
          error: 'MISSING_PRIVILEGES'
        }
      });
    }
    // check the manage_service_account cluster privilege
    else if (request.query.fleetServerSetup) {
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      const {
        has_all_requested: hasAllPrivileges
      } = await esClient.security.hasPrivileges({
        body: {
          cluster: ['manage_service_account']
        }
      });
      if (!hasAllPrivileges) {
        return response.ok({
          body: {
            success: false,
            error: 'MISSING_FLEET_SERVER_SETUP_PRIVILEGES'
          }
        });
      }
    }
    return response.ok({
      body: {
        success: true
      }
    });
  }
};
exports.getCheckPermissionsHandler = getCheckPermissionsHandler;
const generateServiceTokenHandler = async (context, request, response) => {
  // Generate the fleet server service token as the current user as the internal user do not have the correct permissions
  const esClient = (await context.core).elasticsearch.client.asCurrentUser;
  const serviceAccount = request.body.remote ? 'fleet-server-remote' : 'fleet-server';
  _services.appContextService.getLogger().debug(`Creating service token for account elastic/${serviceAccount}`);
  try {
    const tokenResponse = await esClient.transport.request({
      method: 'POST',
      path: `_security/service/elastic/${serviceAccount}/credential/token/token-${Date.now()}`
    });
    if (tokenResponse.created && tokenResponse.token) {
      const body = tokenResponse.token;
      return response.ok({
        body
      });
    } else {
      const error = new _errors.GenerateServiceTokenError('Unable to generate service token');
      return (0, _errors.defaultFleetErrorHandler)({
        error,
        response
      });
    }
  } catch (e) {
    const error = new _errors.GenerateServiceTokenError(e);
    return (0, _errors.defaultFleetErrorHandler)({
      error,
      response
    });
  }
};
exports.generateServiceTokenHandler = generateServiceTokenHandler;
const serviceTokenBodyValidation = (data, validationResult) => {
  const {
    ok
  } = validationResult;
  if (!data) {
    return ok({
      remote: false
    });
  }
  const {
    remote
  } = data;
  return ok({
    remote
  });
};
const registerRoutes = router => {
  router.versioned.get({
    path: _constants.APP_API_ROUTES.CHECK_PERMISSIONS_PATTERN
  }).addVersion({
    version: _constants2.API_VERSIONS.public.v1,
    validate: {
      request: _types.CheckPermissionsRequestSchema
    }
  }, getCheckPermissionsHandler);
  router.versioned.post({
    path: _constants.APP_API_ROUTES.GENERATE_SERVICE_TOKEN_PATTERN,
    fleetAuthz: {
      fleet: {
        all: true
      }
    }
  }).addVersion({
    version: _constants2.API_VERSIONS.public.v1,
    validate: {
      request: {
        body: serviceTokenBodyValidation
      }
    }
  }, generateServiceTokenHandler);
  router.versioned.post({
    path: _constants.APP_API_ROUTES.GENERATE_SERVICE_TOKEN_PATTERN_DEPRECATED,
    fleetAuthz: {
      fleet: {
        all: true
      }
    }
  }).addVersion({
    version: _constants2.API_VERSIONS.public.v1,
    validate: {}
  }, generateServiceTokenHandler);
};
exports.registerRoutes = registerRoutes;