"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getNormalizedInputs = exports.getNormalizedDataStreams = void 0;
exports.isInputOnlyPolicyTemplate = isInputOnlyPolicyTemplate;
exports.isIntegrationPolicyTemplate = isIntegrationPolicyTemplate;
exports.packageHasNoPolicyTemplates = packageHasNoPolicyTemplates;
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DATA_STREAM_DATASET_VAR = {
  name: _constants.DATASET_VAR_NAME,
  type: 'text',
  title: 'Dataset name',
  description: "Set the name for your dataset. Once selected a dataset cannot be changed without creating a new integration policy. You can't use `-` in the name of a dataset and only valid characters for [Elasticsearch index names](https://www.elastic.co/guide/en/elasticsearch/reference/current/docs-index_.html) are permitted.\n",
  multi: false,
  required: true,
  show_user: true
};
function packageHasNoPolicyTemplates(packageInfo) {
  return !packageInfo.policy_templates || packageInfo.policy_templates.length === 0 || !packageInfo.policy_templates.find(policyTemplate => isInputOnlyPolicyTemplate(policyTemplate) || policyTemplate.inputs && policyTemplate.inputs.length > 0);
}
function isInputOnlyPolicyTemplate(policyTemplate) {
  return 'input' in policyTemplate;
}
function isIntegrationPolicyTemplate(policyTemplate) {
  return !isInputOnlyPolicyTemplate(policyTemplate);
}
const getNormalizedInputs = policyTemplate => {
  if (isIntegrationPolicyTemplate(policyTemplate)) {
    return policyTemplate.inputs || [];
  }
  const input = {
    type: policyTemplate.input,
    title: policyTemplate.title,
    description: policyTemplate.description
  };
  return [input];
};
exports.getNormalizedInputs = getNormalizedInputs;
const getNormalizedDataStreams = (packageInfo, datasetName) => {
  if (packageInfo.type !== 'input') {
    return packageInfo.data_streams || [];
  }
  const policyTemplates = packageInfo.policy_templates;
  if (!policyTemplates || policyTemplates.length === 0) {
    return [];
  }
  return policyTemplates.map(policyTemplate => {
    const dataStream = {
      type: policyTemplate.type,
      dataset: datasetName || createDefaultDatasetName(packageInfo, policyTemplate),
      title: policyTemplate.title + ' Dataset',
      release: packageInfo.release || 'ga',
      package: packageInfo.name,
      path: packageInfo.name,
      elasticsearch: packageInfo.elasticsearch || {},
      streams: [{
        input: policyTemplate.input,
        vars: addDatasetVarIfNotPresent(policyTemplate.vars, policyTemplate.name),
        template_path: policyTemplate.template_path,
        title: policyTemplate.title,
        description: policyTemplate.title,
        enabled: true
      }]
    };
    if (packageInfo.type === 'input') {
      dataStream.elasticsearch = {
        ...dataStream.elasticsearch,
        ...{
          dynamic_dataset: true,
          dynamic_namespace: true
        }
      };
    }
    return dataStream;
  });
};

// Input only packages must provide a dataset name in order to differentiate their data streams
// here we add the dataset var if it is not defined in the package already.
exports.getNormalizedDataStreams = getNormalizedDataStreams;
const addDatasetVarIfNotPresent = (vars, datasetName) => {
  const newVars = vars !== null && vars !== void 0 ? vars : [];
  const isDatasetAlreadyAdded = newVars.find(varEntry => varEntry.name === DATA_STREAM_DATASET_VAR.name);
  if (isDatasetAlreadyAdded) {
    return newVars;
  } else {
    return [...newVars, {
      ...DATA_STREAM_DATASET_VAR,
      ...(datasetName && {
        default: datasetName
      })
    }];
  }
};
const createDefaultDatasetName = (packageInfo, policyTemplate) => packageInfo.name + '.' + policyTemplate.name;