"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSearchRoute = registerSearchRoute;
var _configSchema = require("@kbn/config-schema");
var _lib = require("@kbn/search-index-documents/lib");
var _constants = require("../../../common/constants");
var _error_codes = require("../../../common/types/error_codes");
var _create_error = require("../../utils/create_error");
var _elasticsearch_error_handler = require("../../utils/elasticsearch_error_handler");
var _identify_exceptions = require("../../utils/identify_exceptions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerSearchRoute({
  router,
  log
}) {
  router.post({
    path: '/internal/enterprise_search/indices/{index_name}/search',
    validate: {
      body: _configSchema.schema.object({
        searchQuery: _configSchema.schema.string({
          defaultValue: ''
        })
      }),
      params: _configSchema.schema.object({
        index_name: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        page: _configSchema.schema.number({
          defaultValue: 0,
          min: 0
        }),
        size: _configSchema.schema.number({
          defaultValue: _constants.ENTERPRISE_SEARCH_DOCUMENTS_DEFAULT_DOC_COUNT,
          min: 0
        })
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const indexName = decodeURIComponent(request.params.index_name);
    const searchQuery = request.body.searchQuery;
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      page = 0,
      size = _constants.ENTERPRISE_SEARCH_DOCUMENTS_DEFAULT_DOC_COUNT
    } = request.query;
    const from = page * size;
    try {
      const searchResults = await (0, _lib.fetchSearchResults)(client, indexName, searchQuery, from, size);
      return response.ok({
        body: {
          results: searchResults
        },
        headers: {
          'content-type': 'application/json'
        }
      });
    } catch (error) {
      if ((0, _identify_exceptions.isIndexNotFoundException)(error)) {
        return (0, _create_error.createError)({
          errorCode: _error_codes.ErrorCode.INDEX_NOT_FOUND,
          message: 'Could not found index',
          response,
          statusCode: 404
        });
      }
      throw error;
    }
  }));
}