"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postEvaluateRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _uuid = require("uuid");
var _constants = require("../knowledge_base/constants");
var _build_response = require("../../lib/build_response");
var _common = require("../../schemas/common");
var _constants2 = require("../../../common/constants");
var _post_evaluate = require("../../schemas/evaluate/post_evaluate");
var _evaluation = require("../../lib/model_evaluator/evaluation");
var _execute_custom_llm_chain = require("../../lib/langchain/execute_custom_llm_chain");
var _openai_functions_executor = require("../../lib/langchain/executors/openai_functions_executor");
var _actions_client_llm = require("../../lib/langchain/llm/actions_client_llm");
var _utils = require("../../lib/model_evaluator/output_index/utils");
var _utils2 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * To support additional Agent Executors from the UI, add them to this map
 * and reference your specific AgentExecutor function
 */
const AGENT_EXECUTOR_MAP = {
  DefaultAgentExecutor: _execute_custom_llm_chain.callAgentExecutor,
  OpenAIFunctionsExecutor: _openai_functions_executor.callOpenAIFunctionsExecutor
};
const DEFAULT_SIZE = 20;
const postEvaluateRoute = (router, getElser) => {
  router.post({
    path: _constants2.EVALUATE,
    validate: {
      body: (0, _common.buildRouteValidation)(_post_evaluate.PostEvaluateBody),
      query: (0, _common.buildRouteValidation)(_post_evaluate.PostEvaluatePathQuery)
    }
  }, async (context, request, response) => {
    // TODO: Limit route based on experimental feature
    const logger = (await context.elasticAssistant).logger;
    try {
      var _request$query$models, _request$query$agents;
      const evaluationId = (0, _uuid.v4)();
      const {
        evalModel,
        evaluationType,
        outputIndex,
        datasetName,
        projectName = 'default',
        runName = evaluationId
      } = request.query;
      const {
        dataset: customDataset = [],
        evalPrompt
      } = request.body;
      const connectorIds = ((_request$query$models = request.query.models) === null || _request$query$models === void 0 ? void 0 : _request$query$models.split(',')) || [];
      const agentNames = ((_request$query$agents = request.query.agents) === null || _request$query$agents === void 0 ? void 0 : _request$query$agents.split(',')) || [];
      const dataset = datasetName != null ? await (0, _utils2.fetchLangSmithDataset)(datasetName, logger) : customDataset;
      logger.info('postEvaluateRoute:');
      logger.info(`request.query:\n${JSON.stringify(request.query, null, 2)}`);
      logger.info(`request.body:\n${JSON.stringify(request.body, null, 2)}`);
      logger.info(`Evaluation ID: ${evaluationId}`);
      const totalExecutions = connectorIds.length * agentNames.length * dataset.length;
      logger.info('Creating agents:');
      logger.info(`\tconnectors/models: ${connectorIds.length}`);
      logger.info(`\tagents: ${agentNames.length}`);
      logger.info(`\tdataset: ${dataset.length}`);
      logger.warn(`\ttotal baseline agent executions: ${totalExecutions} `);
      if (totalExecutions > 50) {
        logger.warn(`Total baseline agent executions >= 50! This may take a while, and cost some money...`);
      }

      // Get the actions plugin start contract from the request context for the agents
      const actions = (await context.elasticAssistant).actions;

      // Fetch all connectors from the actions plugin, so we can set the appropriate `llmType` on ActionsClientLlm
      const actionsClient = await actions.getActionsClientWithRequest(request);
      const connectors = await actionsClient.getBulk({
        ids: connectorIds,
        throwIfSystemAction: false
      });

      // Get a scoped esClient for passing to the agents for retrieval, and
      // writing results to the output index
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;

      // Default ELSER model
      const elserId = await getElser(request, (await context.core).savedObjects.getClient());

      // Skeleton request to satisfy `subActionParams` spread in `ActionsClientLlm`
      const skeletonRequest = {
        ...request,
        body: {
          alertsIndexPattern: '',
          allow: [],
          allowReplacement: [],
          params: {
            subAction: 'invokeAI',
            subActionParams: {
              messages: []
            }
          },
          replacements: {},
          size: DEFAULT_SIZE,
          assistantLangChain: true
        }
      };

      // Create an array of executor functions to call in batches
      // One for each connector/model + agent combination
      // Hoist `langChainMessages` so they can be batched by dataset.input in the evaluator
      const agents = [];
      connectorIds.forEach(connectorId => {
        agentNames.forEach(agentName => {
          var _getConnectorName;
          logger.info(`Creating agent: ${connectorId} + ${agentName}`);
          const llmType = (0, _utils2.getLlmType)(connectorId, connectors);
          const connectorName = (_getConnectorName = (0, _utils2.getConnectorName)(connectorId, connectors)) !== null && _getConnectorName !== void 0 ? _getConnectorName : '[unknown connector]';
          const detailedRunName = `${runName} - ${connectorName} + ${agentName}`;
          agents.push({
            agentEvaluator: (langChainMessages, exampleId) => AGENT_EXECUTOR_MAP[agentName]({
              actions,
              assistantLangChain: true,
              connectorId,
              esClient,
              elserId,
              langChainMessages,
              llmType,
              logger,
              request: skeletonRequest,
              kbResource: _constants.ESQL_RESOURCE,
              traceOptions: {
                exampleId,
                projectName,
                runName: detailedRunName,
                evaluationId,
                tags: ['security-assistant-prediction', ...(connectorName != null ? [connectorName] : []), runName],
                tracers: (0, _utils2.getLangSmithTracer)(detailedRunName, exampleId, logger)
              }
            }),
            metadata: {
              connectorName,
              runName: detailedRunName
            }
          });
        });
      });
      logger.info(`Agents created: ${agents.length}`);

      // Evaluator Model is optional to support just running predictions
      const evaluatorModel = evalModel == null || evalModel === '' ? undefined : new _actions_client_llm.ActionsClientLlm({
        actions,
        connectorId: evalModel,
        request: skeletonRequest,
        logger
      });
      const {
        evaluationResults,
        evaluationSummary
      } = await (0, _evaluation.performEvaluation)({
        agentExecutorEvaluators: agents,
        dataset,
        evaluationId,
        evaluatorModel,
        evaluationPrompt: evalPrompt,
        evaluationType,
        logger,
        runName
      });
      logger.info(`Writing evaluation results to index: ${outputIndex}`);
      await (0, _utils.setupEvaluationIndex)({
        esClient,
        index: outputIndex,
        logger
      });
      await (0, _utils.indexEvaluations)({
        esClient,
        evaluationResults,
        evaluationSummary,
        index: outputIndex,
        logger
      });
      return response.ok({
        body: {
          evaluationId,
          success: true
        }
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      const resp = (0, _build_response.buildResponse)(response);
      return resp.error({
        body: {
          success: false,
          error: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.postEvaluateRoute = postEvaluateRoute;