"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useTextBasedQueryLanguage = useTextBasedQueryLanguage;
var _lodash = require("lodash");
var _esQuery = require("@kbn/es-query");
var _react = require("react");
var _public = require("@kbn/saved-search-plugin/public");
var _rxjs = require("rxjs");
var _discover_state_provider = require("../services/discover_state_provider");
var _get_valid_view_mode = require("../utils/get_valid_view_mode");
var _types = require("../../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const MAX_NUM_OF_COLUMNS = 50;
// For ES|QL we want in case of the following commands to display a table view, otherwise display a document view
const TRANSFORMATIONAL_COMMANDS = ['stats', 'project', 'keep'];

/**
 * Hook to take care of text based query language state transformations when a new result is returned
 * If necessary this is setting displayed columns and selected data view
 */
function useTextBasedQueryLanguage({
  dataViews,
  stateContainer
}) {
  const prev = (0, _react.useRef)({
    columns: [],
    query: ''
  });
  const initialFetch = (0, _react.useRef)(true);
  const savedSearch = (0, _discover_state_provider.useSavedSearchInitial)();
  const cleanup = (0, _react.useCallback)(() => {
    if (prev.current.query) {
      // cleanup when it's not a text based query lang
      prev.current = {
        columns: [],
        query: ''
      };
    }
  }, []);
  (0, _react.useEffect)(() => {
    const subscription = stateContainer.dataState.data$.documents$.pipe((0, _rxjs.switchMap)(async next => {
      var _next$result;
      const {
        query,
        recordRawType
      } = next;
      if (!query || next.fetchStatus === _types.FetchStatus.ERROR) {
        return;
      }
      const sendComplete = () => {
        stateContainer.dataState.data$.documents$.next({
          ...next,
          fetchStatus: _types.FetchStatus.COMPLETE
        });
      };
      const {
        index,
        viewMode
      } = stateContainer.appState.getState();
      let nextColumns = [];
      const isTextBasedQueryLang = recordRawType === 'plain' && (0, _esQuery.isOfAggregateQueryType)(query) && ('sql' in query || 'esql' in query);
      const hasResults = Boolean((_next$result = next.result) === null || _next$result === void 0 ? void 0 : _next$result.length);
      let queryHasTransformationalCommands = ('sql' in query);
      if ('esql' in query) {
        TRANSFORMATIONAL_COMMANDS.forEach(command => {
          if (query.esql.toLowerCase().includes(command)) {
            queryHasTransformationalCommands = true;
            return;
          }
        });
      }
      if (isTextBasedQueryLang) {
        const language = (0, _esQuery.getAggregateQueryMode)(query);
        if (next.fetchStatus !== _types.FetchStatus.PARTIAL) {
          return;
        }
        const dataViewObj = stateContainer.internalState.getState().dataView;
        if (hasResults) {
          // check if state needs to contain column transformation due to a different columns in the resultset
          const firstRow = next.result[0];
          const firstRowColumns = Object.keys(firstRow.raw).slice(0, MAX_NUM_OF_COLUMNS);
          if (!queryHasTransformationalCommands) {
            nextColumns = [];
            initialFetch.current = false;
          } else {
            nextColumns = firstRowColumns;
            if (initialFetch.current && !prev.current.columns.length && Boolean((dataViewObj === null || dataViewObj === void 0 ? void 0 : dataViewObj.id) === index)) {
              prev.current.columns = firstRowColumns;
            }
          }
        }
        const addColumnsToState = !(0, _lodash.isEqual)(nextColumns, prev.current.columns);
        const queryChanged = query[language] !== prev.current.query;
        // no need to reset index to state if it hasn't changed
        const addDataViewToState = Boolean((dataViewObj === null || dataViewObj === void 0 ? void 0 : dataViewObj.id) !== index);
        if (!queryChanged || !addDataViewToState && !addColumnsToState) {
          sendComplete();
          return;
        }
        if (queryChanged) {
          prev.current.query = query[language];
          prev.current.columns = nextColumns;
        }
        const nextState = {
          ...(addDataViewToState && {
            index: dataViewObj.id
          }),
          ...((addColumnsToState || queryChanged) && {
            columns: nextColumns
          }),
          ...(viewMode === _public.VIEW_MODE.AGGREGATED_LEVEL && {
            viewMode: (0, _get_valid_view_mode.getValidViewMode)({
              viewMode,
              isTextBasedQueryMode: true
            })
          })
        };
        await stateContainer.appState.replaceUrlState(nextState);
        sendComplete();
      } else {
        // cleanup for a "regular" query
        cleanup();
      }
    })).subscribe();
    return () => {
      // cleanup for e.g. when savedSearch is switched
      cleanup();
      subscription.unsubscribe();
    };
  }, [dataViews, stateContainer, savedSearch, cleanup]);
}