"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IndexPatternsFetcher = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _utils = require("../../common/data_views/utils");
var _lib = require("./lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

class IndexPatternsFetcher {
  constructor(elasticsearchClient, allowNoIndices = false, rollupsEnabled = false) {
    (0, _defineProperty2.default)(this, "elasticsearchClient", void 0);
    (0, _defineProperty2.default)(this, "allowNoIndices", void 0);
    (0, _defineProperty2.default)(this, "rollupsEnabled", void 0);
    this.elasticsearchClient = elasticsearchClient;
    this.allowNoIndices = allowNoIndices;
    this.rollupsEnabled = rollupsEnabled;
  }

  /**
   *  Get a list of field objects for an index pattern that may contain wildcards
   *
   *  @param {Object} [options]
   *  @property {String} options.pattern The index pattern
   *  @property {Number} options.metaFields The list of underscore prefixed fields that should
   *                                        be left in the field list (all others are removed).
   *  @return {Promise<Array<Fields>>}
   */
  async getFieldsForWildcard(options) {
    const {
      pattern,
      metaFields = [],
      fieldCapsOptions,
      type,
      rollupIndex,
      indexFilter,
      allowHidden
    } = options;
    const allowNoIndices = fieldCapsOptions ? fieldCapsOptions.allow_no_indices : this.allowNoIndices;
    const expandWildcards = allowHidden ? 'all' : 'open';
    const fieldCapsResponse = await (0, _lib.getFieldCapabilities)({
      callCluster: this.elasticsearchClient,
      indices: pattern,
      metaFields,
      fieldCapsOptions: {
        allow_no_indices: allowNoIndices,
        include_unmapped: fieldCapsOptions === null || fieldCapsOptions === void 0 ? void 0 : fieldCapsOptions.includeUnmapped
      },
      indexFilter,
      fields: options.fields || ['*'],
      expandWildcards
    });
    if (this.rollupsEnabled && type === 'rollup' && rollupIndex) {
      const rollupFields = [];
      const capabilityCheck = (0, _lib.getCapabilitiesForRollupIndices)(await this.elasticsearchClient.rollup.getRollupIndexCaps({
        index: rollupIndex
      }))[rollupIndex];
      if (capabilityCheck.error) {
        throw new Error(capabilityCheck.error);
      }
      const rollupIndexCapabilities = capabilityCheck.aggs;
      const fieldCapsResponseObj = (0, _lodash.keyBy)(fieldCapsResponse.fields, 'name');
      // Keep meta fields
      metaFields.forEach(field => fieldCapsResponseObj[field] && rollupFields.push(fieldCapsResponseObj[field]));
      return {
        fields: (0, _lib.mergeCapabilitiesWithFields)(rollupIndexCapabilities, fieldCapsResponseObj, rollupFields),
        indices: fieldCapsResponse.indices
      };
    }
    return fieldCapsResponse;
  }

  /**
   * Get existing index pattern list by providing string array index pattern list.
   * @param indices - index pattern list
   * @returns index pattern list of index patterns that match indices
   */
  async getExistingIndices(indices) {
    const indicesObs = indices.map(pattern => {
      // when checking a negative pattern, check if the positive pattern exists
      const indexToQuery = pattern.trim().startsWith('-') ? pattern.trim().substring(1) : pattern.trim();
      return (0, _rxjs.defer)(() => (0, _rxjs.from)(this.getFieldsForWildcard({
        // check one field to keep request fast/small
        fields: ['_id'],
        pattern: indexToQuery
      })));
    });
    return new Promise(resolve => {
      (0, _utils.rateLimitingForkJoin)(indicesObs, 3, {
        fields: [],
        indices: []
      }).subscribe(value => {
        resolve(value.map(v => v.indices.length > 0));
      });
    }).then(allPatterns => indices.filter((pattern, i, self) => self.indexOf(pattern) === i && allPatterns[i])).catch(() => indices);
  }
}
exports.IndexPatternsFetcher = IndexPatternsFetcher;