"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerGetApmServicesListFunction = registerGetApmServicesListFunction;
var _datemath = _interopRequireDefault(require("@elastic/datemath"));
var _i18n = require("@kbn/i18n");
var _document_type = require("../../common/document_type");
var _environment_filter_values = require("../../common/environment_filter_values");
var _rollup = require("../../common/rollup");
var _service_health_status = require("../../common/service_health_status");
var _get_apm_alerts_client = require("../lib/helpers/get_apm_alerts_client");
var _get_ml_client = require("../lib/helpers/get_ml_client");
var _get_random_sampler = require("../lib/helpers/get_random_sampler");
var _get_services_items = require("../routes/services/get_services/get_services_items");
var _non_empty_string_ref = require("../utils/non_empty_string_ref");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerGetApmServicesListFunction({
  apmEventClient,
  resources,
  registerFunction
}) {
  registerFunction({
    name: 'get_apm_services_list',
    contexts: ['apm'],
    description: `Gets a list of services`,
    descriptionForUser: _i18n.i18n.translate('xpack.apm.observabilityAiAssistant.functions.registerGetApmServicesList.descriptionForUser', {
      defaultMessage: `Gets the list of monitored services, their health status, and alerts.`
    }),
    parameters: {
      type: 'object',
      additionalProperties: false,
      properties: {
        'service.environment': {
          ..._non_empty_string_ref.NON_EMPTY_STRING,
          description: 'Optionally filter the services by the environments that they are running in'
        },
        start: {
          ..._non_empty_string_ref.NON_EMPTY_STRING,
          description: 'The start of the time range, in Elasticsearch date math, like `now`.'
        },
        end: {
          ..._non_empty_string_ref.NON_EMPTY_STRING,
          description: 'The end of the time range, in Elasticsearch date math, like `now-24h`.'
        },
        healthStatus: {
          type: 'array',
          description: 'Filter service list by health status',
          additionalProperties: false,
          additionalItems: false,
          items: {
            type: 'string',
            enum: [_service_health_status.ServiceHealthStatus.unknown, _service_health_status.ServiceHealthStatus.healthy, _service_health_status.ServiceHealthStatus.warning, _service_health_status.ServiceHealthStatus.critical]
          }
        }
      },
      required: ['start', 'end']
    }
  }, async ({
    arguments: args
  }, signal) => {
    var _datemath$parse, _datemath$parse2;
    const {
      healthStatus
    } = args;
    const [apmAlertsClient, mlClient, randomSampler] = await Promise.all([(0, _get_apm_alerts_client.getApmAlertsClient)(resources), (0, _get_ml_client.getMlClient)(resources), (0, _get_random_sampler.getRandomSampler)({
      security: resources.plugins.security,
      probability: 1,
      request: resources.request
    })]);
    const start = (_datemath$parse = _datemath.default.parse(args.start)) === null || _datemath$parse === void 0 ? void 0 : _datemath$parse.valueOf();
    const end = (_datemath$parse2 = _datemath.default.parse(args.end)) === null || _datemath$parse2 === void 0 ? void 0 : _datemath$parse2.valueOf();
    const serviceItems = await (0, _get_services_items.getServicesItems)({
      apmAlertsClient,
      apmEventClient,
      documentType: _document_type.ApmDocumentType.TransactionMetric,
      start,
      end,
      environment: args['service.environment'] || _environment_filter_values.ENVIRONMENT_ALL.value,
      kuery: '',
      logger: resources.logger,
      randomSampler,
      rollupInterval: _rollup.RollupInterval.OneMinute,
      serviceGroup: null,
      mlClient,
      useDurationSummary: false
    });
    let mappedItems = serviceItems.items.map(item => {
      var _item$alertsCount, _item$healthStatus;
      return {
        'service.name': item.serviceName,
        'agent.name': item.agentName,
        alertsCount: (_item$alertsCount = item.alertsCount) !== null && _item$alertsCount !== void 0 ? _item$alertsCount : 0,
        healthStatus: (_item$healthStatus = item.healthStatus) !== null && _item$healthStatus !== void 0 ? _item$healthStatus : _service_health_status.ServiceHealthStatus.unknown,
        'service.environment': item.environments,
        'transaction.type': item.transactionType
      };
    });
    if (healthStatus && healthStatus.length) {
      mappedItems = mappedItems.filter(item => healthStatus.includes(item.healthStatus));
    }
    return {
      content: mappedItems
    };
  });
}