/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { __awaiter, __generator, __read, __spreadArray } from "tslib";
import { diag, } from '@opentelemetry/api';
import { isObservableInstrument } from '../Instruments';
import { BatchObservableResultImpl, ObservableResultImpl, } from '../ObservableResult';
import { callWithTimeout, PromiseAllSettled, isPromiseAllSettledRejectionResult, setEquals, } from '../utils';
/**
 * An internal interface for managing ObservableCallbacks.
 *
 * Every registered callback associated with a set of instruments are be evaluated
 * exactly once during collection prior to reading data for that instrument.
 */
var ObservableRegistry = /** @class */ (function () {
    function ObservableRegistry() {
        this._callbacks = [];
        this._batchCallbacks = [];
    }
    ObservableRegistry.prototype.addCallback = function (callback, instrument) {
        var idx = this._findCallback(callback, instrument);
        if (idx >= 0) {
            return;
        }
        this._callbacks.push({ callback: callback, instrument: instrument });
    };
    ObservableRegistry.prototype.removeCallback = function (callback, instrument) {
        var idx = this._findCallback(callback, instrument);
        if (idx < 0) {
            return;
        }
        this._callbacks.splice(idx, 1);
    };
    ObservableRegistry.prototype.addBatchCallback = function (callback, instruments) {
        // Create a set of unique instruments.
        var observableInstruments = new Set(instruments.filter(isObservableInstrument));
        if (observableInstruments.size === 0) {
            diag.error('BatchObservableCallback is not associated with valid instruments', instruments);
            return;
        }
        var idx = this._findBatchCallback(callback, observableInstruments);
        if (idx >= 0) {
            return;
        }
        this._batchCallbacks.push({ callback: callback, instruments: observableInstruments });
    };
    ObservableRegistry.prototype.removeBatchCallback = function (callback, instruments) {
        // Create a set of unique instruments.
        var observableInstruments = new Set(instruments.filter(isObservableInstrument));
        var idx = this._findBatchCallback(callback, observableInstruments);
        if (idx < 0) {
            return;
        }
        this._batchCallbacks.splice(idx, 1);
    };
    /**
     * @returns a promise of rejected reasons for invoking callbacks.
     */
    ObservableRegistry.prototype.observe = function (collectionTime, timeoutMillis) {
        return __awaiter(this, void 0, void 0, function () {
            var callbackFutures, batchCallbackFutures, results, rejections;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        callbackFutures = this._observeCallbacks(collectionTime, timeoutMillis);
                        batchCallbackFutures = this._observeBatchCallbacks(collectionTime, timeoutMillis);
                        return [4 /*yield*/, PromiseAllSettled(__spreadArray(__spreadArray([], __read(callbackFutures), false), __read(batchCallbackFutures), false))];
                    case 1:
                        results = _a.sent();
                        rejections = results
                            .filter(isPromiseAllSettledRejectionResult)
                            .map(function (it) { return it.reason; });
                        return [2 /*return*/, rejections];
                }
            });
        });
    };
    ObservableRegistry.prototype._observeCallbacks = function (observationTime, timeoutMillis) {
        var _this = this;
        return this._callbacks.map(function (_a) {
            var callback = _a.callback, instrument = _a.instrument;
            return __awaiter(_this, void 0, void 0, function () {
                var observableResult, callPromise;
                return __generator(this, function (_b) {
                    switch (_b.label) {
                        case 0:
                            observableResult = new ObservableResultImpl(instrument._descriptor);
                            callPromise = Promise.resolve(callback(observableResult));
                            if (timeoutMillis != null) {
                                callPromise = callWithTimeout(callPromise, timeoutMillis);
                            }
                            return [4 /*yield*/, callPromise];
                        case 1:
                            _b.sent();
                            instrument._metricStorages.forEach(function (metricStorage) {
                                metricStorage.record(observableResult._buffer, observationTime);
                            });
                            return [2 /*return*/];
                    }
                });
            });
        });
    };
    ObservableRegistry.prototype._observeBatchCallbacks = function (observationTime, timeoutMillis) {
        var _this = this;
        return this._batchCallbacks.map(function (_a) {
            var callback = _a.callback, instruments = _a.instruments;
            return __awaiter(_this, void 0, void 0, function () {
                var observableResult, callPromise;
                return __generator(this, function (_b) {
                    switch (_b.label) {
                        case 0:
                            observableResult = new BatchObservableResultImpl();
                            callPromise = Promise.resolve(callback(observableResult));
                            if (timeoutMillis != null) {
                                callPromise = callWithTimeout(callPromise, timeoutMillis);
                            }
                            return [4 /*yield*/, callPromise];
                        case 1:
                            _b.sent();
                            instruments.forEach(function (instrument) {
                                var buffer = observableResult._buffer.get(instrument);
                                if (buffer == null) {
                                    return;
                                }
                                instrument._metricStorages.forEach(function (metricStorage) {
                                    metricStorage.record(buffer, observationTime);
                                });
                            });
                            return [2 /*return*/];
                    }
                });
            });
        });
    };
    ObservableRegistry.prototype._findCallback = function (callback, instrument) {
        return this._callbacks.findIndex(function (record) {
            return record.callback === callback && record.instrument === instrument;
        });
    };
    ObservableRegistry.prototype._findBatchCallback = function (callback, instruments) {
        return this._batchCallbacks.findIndex(function (record) {
            return (record.callback === callback &&
                setEquals(record.instruments, instruments));
        });
    };
    return ObservableRegistry;
}());
export { ObservableRegistry };
//# sourceMappingURL=ObservableRegistry.js.map