/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { TraceFlags } from '@opentelemetry/api';
import { internal, ExportResultCode, globalErrorHandler, BindOnceFuture } from '@opentelemetry/core';
/**
 * An implementation of the {@link SpanProcessor} that converts the {@link Span}
 * to {@link ReadableSpan} and passes it to the configured exporter.
 *
 * Only spans that are sampled are converted.
 */
var SimpleSpanProcessor = /** @class */ (function () {
    function SimpleSpanProcessor(_exporter) {
        this._exporter = _exporter;
        this._shutdownOnce = new BindOnceFuture(this._shutdown, this);
    }
    SimpleSpanProcessor.prototype.forceFlush = function () {
        // do nothing as all spans are being exported without waiting
        return Promise.resolve();
    };
    // does nothing.
    SimpleSpanProcessor.prototype.onStart = function (_span, _parentContext) { };
    SimpleSpanProcessor.prototype.onEnd = function (span) {
        if (this._shutdownOnce.isCalled) {
            return;
        }
        if ((span.spanContext().traceFlags & TraceFlags.SAMPLED) === 0) {
            return;
        }
        internal._export(this._exporter, [span]).then(function (result) {
            var _a;
            if (result.code !== ExportResultCode.SUCCESS) {
                globalErrorHandler((_a = result.error) !== null && _a !== void 0 ? _a : new Error("SimpleSpanProcessor: span export failed (status " + result + ")"));
            }
        }).catch(function (error) {
            globalErrorHandler(error);
        });
    };
    SimpleSpanProcessor.prototype.shutdown = function () {
        return this._shutdownOnce.call();
    };
    SimpleSpanProcessor.prototype._shutdown = function () {
        return this._exporter.shutdown();
    };
    return SimpleSpanProcessor;
}());
export { SimpleSpanProcessor };
//# sourceMappingURL=SimpleSpanProcessor.js.map