"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Data = void 0;
var _d = _interopRequireDefault(require("d3"));
var _lodash = _interopRequireDefault(require("lodash"));
var _inject_zeros = require("../components/zero_injection/inject_zeros");
var _ordered_x_keys = require("../components/zero_injection/ordered_x_keys");
var _labels = require("../components/labels/labels");
var _services = require("../../services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

// X axis and split series values in a data table can sometimes be objects,
// e.g. when working with date ranges. d3 casts all ordinal values to strings
// which is a problem for these objects because they just return `[object Object]`
// and thus all map to the same value.
// This little helper overwrites the toString method of an object and keeps it the
// same otherwise - allowing d3 to correctly work with the values.
class D3MappableObject {
  constructor(data) {
    for (const key in data) {
      if (data.hasOwnProperty(key)) {
        this[key] = data[key];
      }
    }
  }
  toString() {
    return JSON.stringify(this);
  }
}

/**
 * Provides an API for pulling values off the data
 * and calculating values using the data
 *
 * @class Data
 * @constructor
 * @param data {Object} Elasticsearch query results
 * @param attr {Object|*} Visualization options
 */
class Data {
  constructor(data, uiState, createColorLookupFunction) {
    this.uiState = uiState;
    this.createColorLookupFunction = createColorLookupFunction;
    this.data = this.copyDataObj(data);
    this.type = this.getDataType();
    this.labels = this._getLabels(this.data);
    this.color = this.labels ? createColorLookupFunction(this.labels, uiState.get('vis.colors')) : undefined;
    this._normalizeOrdered();
  }
  copyDataObj(data) {
    const copyChart = data => {
      const newData = {};
      Object.keys(data).forEach(key => {
        if (key === 'xAxisOrderedValues') {
          newData[key] = data[key].map(val => {
            if (typeof val === 'object') {
              return new D3MappableObject(val);
            }
            return val;
          });
        } else if (key === 'series') {
          newData[key] = data[key].map(seri => {
            const converter = (0, _services.getFormatService)().deserialize(seri.format);
            const zConverter = (0, _services.getFormatService)().deserialize(seri.zFormat);
            return {
              id: seri.id,
              rawId: seri.rawId,
              label: seri.label,
              zLabel: seri.zLabel,
              values: seri.values.map(val => {
                const newVal = _lodash.default.clone(val);
                newVal.extraMetrics = val.extraMetrics;
                newVal.series = val.series || seri.label;
                if (typeof newVal.x === 'object') {
                  newVal.x = new D3MappableObject(newVal.x);
                }
                return newVal;
              }),
              yAxisFormatter: val => converter.convert(val),
              zAxisFormatter: val => zConverter.convert(val)
            };
          });
        } else {
          newData[key] = data[key];
        }
      });
      const xConverter = (0, _services.getFormatService)().deserialize(newData.xAxisFormat);
      const yConverter = (0, _services.getFormatService)().deserialize(newData.yAxisFormat);
      const zConverter = (0, _services.getFormatService)().deserialize(newData.zAxisFormat);
      newData.xAxisFormatter = val => xConverter.convert(val);
      newData.yAxisFormatter = val => yConverter.convert(val);
      newData.zAxisFormatter = val => zConverter.convert(val);
      return newData;
    };
    if (!data.series) {
      const newData = {};
      Object.keys(data).forEach(key => {
        if (!['rows', 'columns'].includes(key)) {
          newData[key] = data[key];
        } else {
          newData[key] = data[key].map(chart => {
            return copyChart(chart);
          });
        }
      });
      return newData;
    }
    return copyChart(data);
  }
  _getLabels(data) {
    if (this.type === 'series') {
      return (0, _labels.labels)(data);
    }
    return [];
  }
  getDataType() {
    const data = this.getVisData();
    let type;
    data.forEach(function (obj) {
      if (obj.series) {
        type = 'series';
      } else if (obj.slices) {
        type = 'slices';
      }
    });
    return type;
  }

  /**
   * Returns an array of the actual x and y data value objects
   * from data with series keys
   *
   * @method chartData
   * @returns {*} Array of data objects
   */
  chartData() {
    if (!this.data.series) {
      const arr = this.data.rows ? this.data.rows : this.data.columns;
      return _lodash.default.toArray(arr);
    }
    return [this.data];
  }
  shouldBeStacked(seriesConfig) {
    if (!seriesConfig) return false;
    return seriesConfig.mode === 'stacked';
  }
  getStackedSeries(chartConfig, axis, series, first = false) {
    const matchingSeries = [];
    chartConfig.series.forEach((seriArgs, i) => {
      const matchingAxis = seriArgs.valueAxis === axis.axisConfig.get('id') || !seriArgs.valueAxis && first;
      if (matchingAxis && (this.shouldBeStacked(seriArgs) || axis.axisConfig.get('scale.stacked'))) {
        matchingSeries.push(series[i]);
      }
    });
    return matchingSeries;
  }
  stackChartData(handler, data, chartConfig) {
    const stackedData = {};
    handler.valueAxes.forEach((axis, i) => {
      const id = axis.axisConfig.get('id');
      stackedData[id] = this.getStackedSeries(chartConfig, axis, data, i === 0);
      stackedData[id] = this.injectZeros(stackedData[id], handler.visConfig.get('orderBucketsBySum', false));
      axis.axisConfig.set('stackedSeries', stackedData[id].length);
      axis.stack(_lodash.default.map(stackedData[id], 'values'));
    });
    return stackedData;
  }
  stackData(handler) {
    const data = this.data;
    if (data.rows || data.columns) {
      const charts = data.rows ? data.rows : data.columns;
      charts.forEach((chart, i) => {
        this.stackChartData(handler, chart.series, handler.visConfig.get(`charts[${i}]`));
      });
    } else {
      this.stackChartData(handler, data.series, handler.visConfig.get('charts[0]'));
    }
  }

  /**
   * Returns an array of chart data objects
   *
   * @method getVisData
   * @returns {*} Array of chart data objects
   */
  getVisData() {
    let visData;
    if (this.data.rows) {
      visData = this.data.rows;
    } else if (this.data.columns) {
      visData = this.data.columns;
    } else {
      visData = [this.data];
    }
    return visData;
  }

  /**
   * Get attributes off the data, e.g. `tooltipFormatter` or `xAxisFormatter`
   * pulls the value off the first item in the array
   * these values are typically the same between data objects of the same chart
   * TODO: May need to verify this or refactor
   *
   * @method get
   * @param thing {String} Data object key
   * @returns {*} Data object value
   */
  get(thing, def) {
    const source = (this.data.rows || this.data.columns || [this.data])[0];
    return _lodash.default.get(source, thing, def);
  }

  /**
   * Returns true if null values are present
   * @returns {*}
   */
  hasNullValues() {
    const chartData = this.chartData();
    return chartData.some(function (chart) {
      return chart.series.some(function (obj) {
        return obj.values.some(function (d) {
          return d.y === null;
        });
      });
    });
  }

  /**
   * Return an array of all value objects
   * Pluck the data.series array from each data object
   * Create an array of all the value objects from the series array
   *
   * @method flatten
   * @returns {Array} Value objects
   */
  flatten() {
    return (0, _lodash.default)(this.chartData()).map('series').flattenDeep().map('values').flattenDeep().value();
  }

  /**
   * Validates that the Y axis min value defined by user input
   * is a number.
   *
   * @param val {Number} Y axis min value
   * @returns {Number} Y axis min value
   */
  validateUserDefinedYMin(val) {
    if (!_lodash.default.isNumber(val)) {
      throw new Error('validateUserDefinedYMin expects a number');
    }
    return val;
  }

  /**
   * Helper function for getNames
   * Returns an array of objects with a name (key) value and an index value.
   * The index value allows us to sort the names in the correct nested order.
   *
   * @method returnNames
   * @param array {Array} Array of data objects
   * @param index {Number} Number of times the object is nested
   * @param columns {Object} Contains name formatter information
   * @returns {Array} Array of labels (strings)
   */
  returnNames(array, index, columns) {
    const names = [];
    const self = this;
    _lodash.default.forEach(array, function (obj) {
      names.push({
        label: obj.name,
        values: [obj.rawData],
        index: index
      });
      if (obj.children) {
        const plusIndex = index + 1;
        _lodash.default.forEach(self.returnNames(obj.children, plusIndex, columns), function (namedObj) {
          names.push(namedObj);
        });
      }
    });
    return names;
  }

  /**
   * Flattens hierarchical data into an array of objects with a name and index value.
   * The indexed value determines the order of nesting in the data.
   * Returns an array with names sorted by the index value.
   *
   * @method getNames
   * @param data {Object} Chart data object
   * @param columns {Object} Contains formatter information
   * @returns {Array} Array of names (strings)
   */
  getNames(data, columns) {
    const slices = data.slices;
    if (slices.children) {
      const namedObj = this.returnNames(slices.children, 0, columns);
      return (0, _lodash.default)(namedObj).sortBy(function (obj) {
        return obj.index;
      }).uniqBy(function (d) {
        return d.label;
      }).value();
    }
  }

  /**
   * Inject zeros into the data
   *
   * @method injectZeros
   * @returns {Object} Data object with zeros injected
   */
  injectZeros(data, orderBucketsBySum = false) {
    return (0, _inject_zeros.injectZeros)(data, this.data, orderBucketsBySum);
  }

  /**
   * Returns an array of all x axis values from the data
   *
   * @method xValues
   * @returns {Array} Array of x axis values
   */
  xValues(orderBucketsBySum = false) {
    return (0, _ordered_x_keys.orderXValues)(this.data, orderBucketsBySum);
  }

  /**
   * Return an array of unique labels
   * Currently, only used for vertical bar and line charts,
   * or any data object with series values
   *
   * @method getLabels
   * @returns {Array} Array of labels (strings)
   */
  getLabels() {
    return (0, _labels.labels)(this.data);
  }

  /**
   * Returns a function that does color lookup on labels
   *
   * @method getColorFunc
   * @returns {Function} Performs lookup on string and returns hex color
   */
  getColorFunc() {
    const defaultColors = this.uiState.get('vis.defaultColors');
    const overwriteColors = this.uiState.get('vis.colors');
    const colors = defaultColors ? _lodash.default.defaults({}, overwriteColors, defaultColors) : overwriteColors;
    return this.createColorLookupFunction(this.getLabels(), colors);
  }

  /**
   * ensure that the datas ordered property has a min and max
   * if the data represents an ordered date range.
   *
   * @return {undefined}
   */
  _normalizeOrdered() {
    const data = this.getVisData();
    const self = this;
    data.forEach(function (d) {
      if (!d.ordered || !d.ordered.date) return;
      const missingMin = d.ordered.min == null;
      const missingMax = d.ordered.max == null;
      if (missingMax || missingMin) {
        const extent = _d.default.extent(self.xValues());
        if (missingMin) d.ordered.min = extent[0];
        if (missingMax) d.ordered.max = extent[1];
      }
    });
  }

  /**
   * Calculates min and max values for all map data
   * series.rows is an array of arrays
   * each row is an array of values
   * last value in row array is bucket count
   *
   * @method mapDataExtents
   * @param series {Array} Array of data objects
   * @returns {Array} min and max values
   */
  mapDataExtents(series) {
    const values = _lodash.default.map(series.rows, function (row) {
      return row[row.length - 1];
    });
    return [_lodash.default.min(values), _lodash.default.max(values)];
  }

  /**
   * Get the maximum number of series, considering each chart
   * individually.
   *
   * @return {number} - the largest number of series from all charts
   */
  maxNumberOfSeries() {
    return this.chartData().reduce(function (max, chart) {
      return Math.max(max, chart.series.length);
    }, 0);
  }
}
exports.Data = Data;