"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VegaParser = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = _interopRequireDefault(require("lodash"));
var _vegaSchemaUrlParser = _interopRequireDefault(require("vega-schema-url-parser"));
var _compareVersions = _interopRequireDefault(require("compare-versions"));
var _hjson = _interopRequireDefault(require("hjson"));
var _eui = require("@elastic/eui");
var _uiTheme = require("@kbn/ui-theme");
var _i18n = require("@kbn/i18n");
var _vega = require("vega");
var _vegaLite = require("vega-lite");
var _es_query_parser = require("./es_query_parser");
var _utils = require("./utils");
var _ems_file_parser = require("./ems_file_parser");
var _url_parser = require("./url_parser");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

// Set default single color to match other Kibana visualizations
const defaultColor = (0, _eui.euiPaletteColorBlind)()[0];
const locToDirMap = {
  left: 'row-reverse',
  right: 'row',
  top: 'column-reverse',
  bottom: 'column'
};

// If there is no "%type%" parameter, use this parser
const DEFAULT_PARSER = 'elasticsearch';
class VegaParser {
  constructor(spec, searchAPI, timeCache, filters, getServiceSettings) {
    (0, _defineProperty2.default)(this, "spec", void 0);
    (0, _defineProperty2.default)(this, "hideWarnings", void 0);
    (0, _defineProperty2.default)(this, "restoreSignalValuesOnRefresh", void 0);
    (0, _defineProperty2.default)(this, "error", void 0);
    (0, _defineProperty2.default)(this, "warnings", void 0);
    (0, _defineProperty2.default)(this, "_urlParsers", void 0);
    (0, _defineProperty2.default)(this, "isVegaLite", void 0);
    (0, _defineProperty2.default)(this, "useHover", void 0);
    (0, _defineProperty2.default)(this, "_config", void 0);
    (0, _defineProperty2.default)(this, "useMap", void 0);
    (0, _defineProperty2.default)(this, "renderer", void 0);
    (0, _defineProperty2.default)(this, "tooltips", void 0);
    (0, _defineProperty2.default)(this, "mapConfig", void 0);
    (0, _defineProperty2.default)(this, "vlspec", void 0);
    (0, _defineProperty2.default)(this, "useResize", void 0);
    (0, _defineProperty2.default)(this, "containerDir", void 0);
    (0, _defineProperty2.default)(this, "controlsDir", void 0);
    (0, _defineProperty2.default)(this, "searchAPI", void 0);
    (0, _defineProperty2.default)(this, "getServiceSettings", void 0);
    (0, _defineProperty2.default)(this, "filters", void 0);
    (0, _defineProperty2.default)(this, "timeCache", void 0);
    this.spec = spec;
    this.hideWarnings = false;
    this.error = undefined;
    this.warnings = [];
    this.searchAPI = searchAPI;
    this.getServiceSettings = getServiceSettings;
    this.filters = filters;
    this.timeCache = timeCache;
  }
  async parseAsync() {
    try {
      await this._parseAsync();
    } catch (err) {
      // if we reject current promise, it will use the standard Kibana error handling
      this.error = _utils.Utils.formatErrorToStr(err);
    }
    return this;
  }
  async _parseAsync() {
    if (this.isVegaLite !== undefined) throw new Error();
    if (typeof this.spec === 'string') {
      const spec = _hjson.default.parse(this.spec, {
        legacyRoot: false
      });
      if (!spec.$schema) {
        throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.inputSpecDoesNotSpecifySchemaErrorMessage', {
          defaultMessage: `Your specification requires a {schemaParam} field with a valid URL for
Vega (see {vegaSchemaUrl}) or
Vega-Lite (see {vegaLiteSchemaUrl}).
The URL is an identifier only. Kibana and your browser will never access this URL.`,
          values: {
            schemaParam: '"$schema"',
            vegaLiteSchemaUrl: 'https://vega.github.io/vega-lite/docs/spec.html#top-level',
            vegaSchemaUrl: 'https://vega.github.io/vega/docs/specification/#top-level-specification-properties'
          }
        }));
      }
      this.spec = spec;
    }
    if (!_lodash.default.isPlainObject(this.spec)) {
      throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.invalidVegaSpecErrorMessage', {
        defaultMessage: 'Invalid Vega specification'
      }));
    }
    this.isVegaLite = this.parseSchema(this.spec).isVegaLite;
    this.useHover = !this.isVegaLite;
    this._config = this._parseConfig();
    this.hideWarnings = !!this._config.hideWarnings;
    this._parseBool('restoreSignalValuesOnRefresh', this._config, false);
    this.restoreSignalValuesOnRefresh = this._config.restoreSignalValuesOnRefresh;
    this.useMap = this._config.type === 'map';
    this.renderer = this._config.renderer === 'svg' ? 'svg' : 'canvas';
    this.tooltips = this._parseTooltips();
    this._setDefaultColors();
    this._parseControlPlacement();
    if (this.useMap) {
      this.mapConfig = this._parseMapConfig();
      this.useResize = false;
    }
    await this._resolveDataUrls();
    if (this.isVegaLite) {
      this._compileVegaLite();
    } else {
      this._compileWithAutosize();
    }
  }

  /**
   * Ensure that Vega and Vega-Lite will take the full width of the container unless
   * the user has explicitly disabled this setting by setting it to "none".
   * Also sets the default width to include the padding. This creates the least configuration
   * needed for most cases, with the option to do more.
   */
  _compileWithAutosize() {
    const defaultAutosize = {
      type: 'fit',
      contains: 'padding'
    };
    let autosize = this.spec.autosize;
    let useResize = true;
    if (!this.isVegaLite && autosize && typeof autosize === 'object' && 'signal' in autosize) {
      // Vega supports dynamic autosize information, so we ignore it
      return;
    }
    if (!autosize && typeof autosize !== 'undefined') {
      this._onWarning(_i18n.i18n.translate('visTypeVega.vegaParser.autoSizeDoesNotAllowFalse', {
        defaultMessage: '{autoSizeParam} is enabled, it can only be disabled by setting {autoSizeParam} to {noneParam}',
        values: {
          autoSizeParam: '"autosize"',
          noneParam: '"none"'
        }
      }));
    }
    if (typeof autosize === 'string') {
      useResize = autosize !== 'none';
      autosize = {
        ...defaultAutosize,
        type: autosize
      };
    } else if (typeof autosize === 'object') {
      var _autosize, _autosize2;
      autosize = {
        ...defaultAutosize,
        ...autosize
      };
      useResize = Boolean(((_autosize = autosize) === null || _autosize === void 0 ? void 0 : _autosize.type) && ((_autosize2 = autosize) === null || _autosize2 === void 0 ? void 0 : _autosize2.type) !== 'none');
    } else {
      autosize = defaultAutosize;
    }
    if (useResize && (this.spec.width && this.spec.width !== 'container' || this.spec.height && this.spec.height !== 'container')) {
      this._onWarning(_i18n.i18n.translate('visTypeVega.vegaParser.widthAndHeightParamsAreIgnored', {
        defaultMessage: '{widthParam} and {heightParam} params are ignored because {autoSizeParam} is enabled. Set {autoSizeParam}: {noneParam} to disable',
        values: {
          widthParam: '"width"',
          heightParam: '"height"',
          autoSizeParam: '"autosize"',
          noneParam: '"none"'
        }
      }));
    }
    if (useResize) {
      this.spec.width = 'container';
      this.spec.height = 'container';
    }
    this.spec.autosize = autosize;
    this.useResize = useResize;
  }

  /**
   * Convert VegaLite to Vega spec
   */
  _compileVegaLite() {
    if (!this.useMap) {
      // Compile without warnings to get the normalized spec, this simplifies the autosize detection
      const normalized = (0, _vegaLite.compile)(this.spec, {
        logger: (0, _vega.logger)(_vega.None)
      }).normalized;

      // Vega-Lite allows autosize when there is a single mark or layered chart, but
      // does not allow autosize for other specs.
      if ('mark' in normalized || 'layer' in normalized) {
        this._compileWithAutosize();
      } else {
        this.useResize = false;
        if (normalized.autosize && typeof normalized.autosize !== 'string' && normalized.autosize.type === 'none') {
          this._onWarning(_i18n.i18n.translate('visTypeVega.vegaParser.widthAndHeightParamsAreRequired', {
            defaultMessage: 'Nothing is rendered when {autoSizeParam} is set to {noneParam} while using faceted or repeated {vegaLiteParam} specs. To fix, remove {autoSizeParam} or use {vegaParam}.',
            values: {
              autoSizeParam: '"autosize"',
              noneParam: '"none"',
              vegaLiteParam: 'Vega-Lite',
              vegaParam: 'Vega'
            }
          }));
        }
      }
    }
    this.vlspec = this.spec;
    const vegaLogger = (0, _vega.logger)(_vega.Warn); // note: eslint has a false positive here
    vegaLogger.warn = this._onWarning.bind(this);
    this.spec = (0, _vegaLite.compile)(this.vlspec, {
      logger: vegaLogger
    }).spec;

    // When using Vega-Lite (VL) with the type=map and user did not provid their own projection settings,
    // remove the default projection that was generated by VegaLite compiler.
    if (this.useMap) {
      if (!this.spec || !this.vlspec) return;
      const hasConfig = _lodash.default.isPlainObject(this.vlspec.config);
      if (this.vlspec.config === undefined || hasConfig && !this.vlspec.config.projection) {
        // Assume VL generates spec.projections = an array of exactly one object named 'projection'
        if (!Array.isArray(this.spec.projections) || this.spec.projections.length !== 1 || this.spec.projections[0].name !== 'projection') {
          throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.VLCompilerShouldHaveGeneratedSingleProtectionObjectErrorMessage', {
            defaultMessage: 'Internal error: Vega-Lite compiler should have generated a single projection object'
          }));
        }
        delete this.spec.projections;
      }

      // todo: sizing cleanup might need to be rethought and consolidated
      if (!this.vlspec.width) delete this.spec.width;
      if (!this.vlspec.height) delete this.spec.height;
      if (!this.vlspec.padding && (this.vlspec.config === undefined || hasConfig && !this.vlspec.config.padding)) {
        delete this.spec.padding;
      }
      if (!this.vlspec.autosize && (this.vlspec.config === undefined || hasConfig && !this.vlspec.config.autosize)) {
        delete this.spec.autosize;
      }
    }
  }

  /**
   * Calculate container-direction CSS property for binding placement
   * @private
   */
  _parseControlPlacement() {
    var _this$_config, _this$_config2;
    this.containerDir = (_this$_config = this._config) !== null && _this$_config !== void 0 && _this$_config.controlsLocation ? locToDirMap[this._config.controlsLocation] : undefined;
    if (this.containerDir === undefined) {
      if (this._config && this._config.controlsLocation === undefined) {
        this.containerDir = 'column';
      } else {
        throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.unrecognizedControlsLocationValueErrorMessage', {
          defaultMessage: 'Unrecognized {controlsLocationParam} value. Expecting one of [{locToDirMap}]',
          values: {
            locToDirMap: `"${Object.keys(locToDirMap).join('", "')}"`,
            controlsLocationParam: 'controlsLocation'
          }
        }));
      }
    }
    const dir = (_this$_config2 = this._config) === null || _this$_config2 === void 0 ? void 0 : _this$_config2.controlsDirection;
    if (dir !== undefined && dir !== 'horizontal' && dir !== 'vertical') {
      throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.unrecognizedDirValueErrorMessage', {
        defaultMessage: 'Unrecognized {dirParam} value. Expecting one of [{expectedValues}]',
        values: {
          expectedValues: '"horizontal", "vertical"',
          dirParam: 'dir'
        }
      }));
    }
    this.controlsDir = dir === 'horizontal' ? 'row' : 'column';
  }

  /**
   * Parse {config: kibana: {...}} portion of the Vega spec (or root-level _hostConfig for backward compat)
   * @returns {object} kibana config
   * @private
   */
  _parseConfig() {
    let result = null;
    if (this.spec) {
      if (this.spec._hostConfig !== undefined) {
        result = this.spec._hostConfig;
        delete this.spec._hostConfig;
        if (!_lodash.default.isPlainObject(result)) {
          throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.hostConfigValueTypeErrorMessage', {
            defaultMessage: 'If present, {configName} must be an object',
            values: {
              configName: '"_hostConfig"'
            }
          }));
        }
        this._onWarning(_i18n.i18n.translate('visTypeVega.vegaParser.hostConfigIsDeprecatedWarningMessage', {
          defaultMessage: '{deprecatedConfigName} has been deprecated. Use {newConfigName} instead.',
          values: {
            deprecatedConfigName: '"_hostConfig"',
            newConfigName: 'config.kibana'
          }
        }));
      }
      if (_lodash.default.isPlainObject(this.spec.config) && this.spec.config.kibana !== undefined) {
        result = this.spec.config.kibana;
        delete this.spec.config.kibana;
        if (!_lodash.default.isPlainObject(result)) {
          throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.kibanaConfigValueTypeErrorMessage', {
            defaultMessage: 'If present, {configName} must be an object',
            values: {
              configName: 'config.kibana'
            }
          }));
        }
      }
    }
    return result || {};
  }
  _parseTooltips() {
    var _this$_config3;
    if (this._config && this._config.tooltips === false) {
      return false;
    }
    const result = ((_this$_config3 = this._config) === null || _this$_config3 === void 0 ? void 0 : _this$_config3.tooltips) || {};
    if (result.position === undefined) {
      result.position = 'top';
    } else if (['top', 'right', 'bottom', 'left'].indexOf(result.position) === -1) {
      throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.unexpectedValueForPositionConfigurationErrorMessage', {
        defaultMessage: 'Unexpected value for the {configurationName} configuration',
        values: {
          configurationName: 'result.position'
        }
      }));
    }
    if (result.padding === undefined) {
      result.padding = 16;
    } else if (typeof result.padding !== 'number') {
      throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.paddingConfigValueTypeErrorMessage', {
        defaultMessage: '{configName} is expected to be a number',
        values: {
          configName: 'config.kibana.result.padding'
        }
      }));
    }
    if (result.textTruncate === undefined) {
      result.textTruncate = false;
    } else if (typeof result.textTruncate !== 'boolean') {
      throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.textTruncateConfigValueTypeErrorMessage', {
        defaultMessage: '{configName} is expected to be a boolean',
        values: {
          configName: 'textTruncate'
        }
      }));
    }
    if (result.centerOnMark === undefined) {
      // if mark's width & height is less than this value, center on it
      result.centerOnMark = 50;
    } else if (typeof result.centerOnMark === 'boolean') {
      result.centerOnMark = result.centerOnMark ? Number.MAX_VALUE : -1;
    } else if (typeof result.centerOnMark !== 'number') {
      throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.centerOnMarkConfigValueTypeErrorMessage', {
        defaultMessage: '{configName} is expected to be {trueValue}, {falseValue}, or a number',
        values: {
          configName: 'config.kibana.result.centerOnMark',
          trueValue: 'true',
          falseValue: 'false'
        }
      }));
    }
    return result;
  }

  /**
   * Parse map-specific configuration
   * @returns {{mapStyle: *|string, delayRepaint: boolean, latitude: number, longitude: number, zoom, minZoom, maxZoom, zoomControl: *|boolean, maxBounds: *}}
   * @private
   */
  _parseMapConfig() {
    var _this$_config4, _this$_config6;
    const res = {
      delayRepaint: ((_this$_config4 = this._config) === null || _this$_config4 === void 0 ? void 0 : _this$_config4.delayRepaint) === undefined ? true : this._config.delayRepaint
    };
    const validate = (name, isZoom) => {
      const val = this._config ? this._config[name] : undefined;
      if (val !== undefined) {
        const parsed = parseFloat(val);
        if (Number.isFinite(parsed) && (!isZoom || parsed >= 0 && parsed <= 30)) {
          res[name] = parsed;
          return;
        }
        this._onWarning(_i18n.i18n.translate('visTypeVega.vegaParser.someKibanaConfigurationIsNoValidWarningMessage', {
          defaultMessage: '{configName} is not valid',
          values: {
            configName: `config.kibana.${name}`
          }
        }));
      }
      if (!isZoom) res[name] = 0;
    };
    validate(`latitude`, false);
    validate(`longitude`, false);
    validate(`zoom`, true);
    validate(`minZoom`, true);
    validate(`maxZoom`, true);
    this._parseBool('mapStyle', res, true);
    if (res.mapStyle) {
      var _this$_config5;
      res.emsTileServiceId = (_this$_config5 = this._config) === null || _this$_config5 === void 0 ? void 0 : _this$_config5.emsTileServiceId;
    }
    this._parseBool('zoomControl', res, true);
    this._parseBool('scrollWheelZoom', res, false);
    const maxBounds = (_this$_config6 = this._config) === null || _this$_config6 === void 0 ? void 0 : _this$_config6.maxBounds;
    if (maxBounds !== undefined) {
      if (!Array.isArray(maxBounds) || maxBounds.length !== 4 || !maxBounds.every(v => typeof v === 'number' && Number.isFinite(v))) {
        this._onWarning(_i18n.i18n.translate('visTypeVega.vegaParser.maxBoundsValueTypeWarningMessage', {
          defaultMessage: '{maxBoundsConfigName} must be an array with four numbers',
          values: {
            maxBoundsConfigName: 'config.kibana.maxBounds'
          }
        }));
      } else {
        res.maxBounds = maxBounds;
      }
    }
    return res;
  }
  _parseBool(paramName, dstObj, dflt) {
    const val = this._config ? this._config[paramName] : undefined;
    if (val === undefined) {
      dstObj[paramName] = dflt;
    } else if (typeof val !== 'boolean') {
      this._onWarning(_i18n.i18n.translate('visTypeVega.vegaParser.someKibanaParamValueTypeWarningMessage', {
        defaultMessage: '{configName} must be a boolean value',
        values: {
          configName: `config.kibana.${paramName}`
        }
      }));
      dstObj[paramName] = dflt;
    } else {
      dstObj[paramName] = val;
    }
  }

  /**
   * Parse Vega schema element
   * @returns {object} isVegaLite, libVersion
   * @private
   */
  parseSchema(spec) {
    try {
      const schema = (0, _vegaSchemaUrlParser.default)(spec.$schema);
      const isVegaLite = schema.library === 'vega-lite';
      const libVersion = isVegaLite ? _vegaLite.version : _vega.version;
      if ((0, _compareVersions.default)(schema.version, libVersion) > 0) {
        this._onWarning(_i18n.i18n.translate('visTypeVega.vegaParser.notValidLibraryVersionForInputSpecWarningMessage', {
          defaultMessage: 'The input spec uses {schemaLibrary} {schemaVersion}, but current version of {schemaLibrary} is {libraryVersion}.',
          values: {
            schemaLibrary: schema.library,
            schemaVersion: schema.version,
            libraryVersion: libVersion
          }
        }));
      }
      return {
        isVegaLite,
        libVersion
      };
    } catch (e) {
      throw Error(_i18n.i18n.translate('visTypeVega.vegaParser.notValidSchemaForInputSpec', {
        defaultMessage: 'The URL for the JSON "$schema" is incorrect. Correct the URL, then click Update.'
      }));
    }
  }

  /**
   * Replace all instances of ES requests with raw values.
   * Also handle any other type of url: {type: xxx, ...}
   * @private
   */
  async _resolveDataUrls() {
    if (!this._urlParsers) {
      const serviceSettings = await this.getServiceSettings();
      const onWarn = this._onWarning.bind(this);
      this._urlParsers = {
        elasticsearch: new _es_query_parser.EsQueryParser(this.timeCache, this.searchAPI, this.filters, onWarn),
        emsfile: new _ems_file_parser.EmsFileParser(serviceSettings),
        url: new _url_parser.UrlParser(onWarn)
      };
    }
    const pending = {};
    this.searchAPI.resetSearchStats();
    this._findObjectDataUrls(this.spec, obj => {
      const url = obj.url;
      delete obj.url;
      let type = url['%type%'];
      delete url['%type%'];
      if (type === undefined) {
        type = DEFAULT_PARSER;
      }
      const parser = this._urlParsers[type];
      if (parser === undefined) {
        throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.notSupportedUrlTypeErrorMessage', {
          defaultMessage: '{urlObject} is not supported',
          values: {
            urlObject: 'url: {"%type%": "${type}"}'
          }
        }));
      }
      let pendingArr = pending[type];
      if (pendingArr === undefined) {
        pending[type] = pendingArr = [];
      }
      pendingArr.push(parser.parseUrl(obj, url));
    });
    const pendingParsers = Object.keys(pending);
    if (pendingParsers.length > 0) {
      // let each parser populate its data in parallel
      await Promise.all(pendingParsers.map(type => this._urlParsers[type].populateData(pending[type])));
    }
  }

  /**
   * Recursively find and callback every instance of the data.url as an object
   * @param {*} obj current location in the object tree
   * @param {function({object})} onFind Call this function for all url objects
   * @param {string} [key] field name of the current object
   * @private
   */

  _findObjectDataUrls(obj, onFind, key) {
    if (Array.isArray(obj)) {
      for (const elem of obj) {
        this._findObjectDataUrls(elem, onFind, key);
      }
    } else if (_lodash.default.isPlainObject(obj)) {
      if (key === 'data' && _lodash.default.isPlainObject(obj.url)) {
        // Assume that any  "data": {"url": {...}}  is a request for data
        if (obj.values !== undefined || obj.source !== undefined) {
          throw new Error(_i18n.i18n.translate('visTypeVega.vegaParser.dataExceedsSomeParamsUseTimesLimitErrorMessage', {
            defaultMessage: 'Data must not have more than one of {urlParam}, {valuesParam}, and {sourceParam}',
            values: {
              urlParam: '"url"',
              valuesParam: '"values"',
              sourceParam: '"source"'
            }
          }));
        }
        onFind(obj);
      } else if (key === 'data' && typeof obj.url === 'string') {
        const bounds = this.timeCache.getTimeBounds();
        obj.url = obj.url.replaceAll('%timefilter_min%', bounds.min.toString()).replaceAll('%timefilter_max%', bounds.max.toString());
      } else {
        for (const k of Object.keys(obj)) {
          this._findObjectDataUrls(obj[k], onFind, k);
        }
      }
    }
  }

  /**
   * Inject default colors into the spec.config
   * @private
   */
  _setDefaultColors() {
    // Default category coloring to the Elastic color scheme
    this._setDefaultValue({
      scheme: 'elastic'
    }, 'config', 'range', 'category');
    if (this.isVegaLite) {
      // Vega-Lite: set default color, works for fill and strike --  config: { mark:  { color: '#54B399' }}
      this._setDefaultValue(defaultColor, 'config', 'mark', 'color');
    } else {
      var _this$spec;
      // Vega - global mark has very strange behavior, must customize each mark type individually
      // https://github.com/vega/vega/issues/1083
      // Don't set defaults if spec.config.mark.color or fill are set
      if (!((_this$spec = this.spec) !== null && _this$spec !== void 0 && _this$spec.config.mark) || this.spec.config.mark.color === undefined && this.spec.config.mark.fill === undefined) {
        this._setDefaultValue(defaultColor, 'config', 'arc', 'fill');
        this._setDefaultValue(defaultColor, 'config', 'area', 'fill');
        this._setDefaultValue(defaultColor, 'config', 'line', 'stroke');
        this._setDefaultValue(defaultColor, 'config', 'path', 'stroke');
        this._setDefaultValue(defaultColor, 'config', 'rect', 'fill');
        this._setDefaultValue(defaultColor, 'config', 'rule', 'stroke');
        this._setDefaultValue(defaultColor, 'config', 'shape', 'stroke');
        this._setDefaultValue(defaultColor, 'config', 'symbol', 'fill');
        this._setDefaultValue(defaultColor, 'config', 'trail', 'fill');
      }
    }

    // provide right colors for light and dark themes
    this._setDefaultValue(_uiTheme.euiThemeVars.euiColorDarkestShade, 'config', 'title', 'color');
    this._setDefaultValue(_uiTheme.euiThemeVars.euiColorDarkShade, 'config', 'style', 'guide-label', 'fill');
    this._setDefaultValue(_uiTheme.euiThemeVars.euiColorDarkestShade, 'config', 'style', 'guide-title', 'fill');
    this._setDefaultValue(_uiTheme.euiThemeVars.euiColorDarkestShade, 'config', 'style', 'group-title', 'fill');
    this._setDefaultValue(_uiTheme.euiThemeVars.euiColorDarkestShade, 'config', 'style', 'group-subtitle', 'fill');
    this._setDefaultValue(_uiTheme.euiThemeVars.euiColorChartLines, 'config', 'axis', 'tickColor');
    this._setDefaultValue(_uiTheme.euiThemeVars.euiColorChartLines, 'config', 'axis', 'domainColor');
    this._setDefaultValue(_uiTheme.euiThemeVars.euiColorChartLines, 'config', 'axis', 'gridColor');
    this._setDefaultValue('transparent', 'config', 'background');
  }

  /**
   * Set default value if it doesn't exist.
   * Given an object, and an array of fields, ensure that obj.fld1.fld2. ... .fldN is set to value if it doesn't exist.
   * @param {*} value
   * @param {string} fields
   * @private
   */
  _setDefaultValue(value, ...fields) {
    let o = this.spec;
    for (let i = 0; i < fields.length - 1; i++) {
      const field = fields[i];
      const subObj = o[field];
      if (subObj === undefined) {
        o[field] = {};
      } else if (!_lodash.default.isPlainObject(subObj)) {
        return;
      }
      o = o[field];
    }
    const lastField = fields[fields.length - 1];
    if (o[lastField] === undefined) {
      o[lastField] = value;
    }
  }

  /**
   * Add a warning to the warnings array
   * @private
   */
  _onWarning(...args) {
    if (!this.hideWarnings) {
      this.warnings.push(_utils.Utils.formatWarningToStr(args));
      return _utils.Utils.formatWarningToStr(args);
    }
  }
}
exports.VegaParser = VegaParser;