"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateWildcardInput = exports.validatePotentialWildcardInput = exports.validateFilePathInput = exports.isPathValid = exports.hasSimpleExecutableName = exports.WILDCARD_WARNING = exports.OperatingSystem = exports.FILEPATH_WARNING = exports.EntryFieldType = exports.ConditionEntryField = void 0;
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const WILDCARD_WARNING = _i18n.i18n.translate('utils.wildcardWarning', {
  defaultMessage: `Using wildcards can impact Endpoint performance`
});
exports.WILDCARD_WARNING = WILDCARD_WARNING;
const FILEPATH_WARNING = _i18n.i18n.translate('utils.filename.pathWarning', {
  defaultMessage: `Path may be formed incorrectly; verify value`
});
exports.FILEPATH_WARNING = FILEPATH_WARNING;
let ConditionEntryField;
exports.ConditionEntryField = ConditionEntryField;
(function (ConditionEntryField) {
  ConditionEntryField["HASH"] = "process.hash.*";
  ConditionEntryField["PATH"] = "process.executable.caseless";
  ConditionEntryField["SIGNER"] = "process.Ext.code_signature";
})(ConditionEntryField || (exports.ConditionEntryField = ConditionEntryField = {}));
let EntryFieldType;
exports.EntryFieldType = EntryFieldType;
(function (EntryFieldType) {
  EntryFieldType["HASH"] = ".hash.";
  EntryFieldType["EXECUTABLE"] = ".executable.caseless";
  EntryFieldType["PATH"] = ".path";
  EntryFieldType["SIGNER"] = ".Ext.code_signature";
})(EntryFieldType || (exports.EntryFieldType = EntryFieldType = {}));
let OperatingSystem;
exports.OperatingSystem = OperatingSystem;
(function (OperatingSystem) {
  OperatingSystem["LINUX"] = "linux";
  OperatingSystem["MAC"] = "macos";
  OperatingSystem["WINDOWS"] = "windows";
})(OperatingSystem || (exports.OperatingSystem = OperatingSystem = {}));
const validatePotentialWildcardInput = ({
  field = '',
  os,
  value = ''
}) => {
  const textInput = value.trim();
  if (field === 'file.path.text') {
    return validateFilePathInput({
      os,
      value: textInput
    });
  }
  return validateWildcardInput(textInput);
};
exports.validatePotentialWildcardInput = validatePotentialWildcardInput;
const validateFilePathInput = ({
  os,
  value
}) => {
  const isValidFilePath = isPathValid({
    os,
    field: 'file.path.text',
    type: 'wildcard',
    value
  });
  const hasSimpleFileName = hasSimpleExecutableName({
    os,
    type: 'wildcard',
    value
  });
  if (!value.length) {
    return FILEPATH_WARNING;
  }
  if (isValidFilePath) {
    if (hasSimpleFileName !== undefined && !hasSimpleFileName) {
      return WILDCARD_WARNING;
    }
  } else {
    return FILEPATH_WARNING;
  }
};
exports.validateFilePathInput = validateFilePathInput;
const validateWildcardInput = value => {
  if (/[*?]/.test(value !== null && value !== void 0 ? value : '')) {
    return WILDCARD_WARNING;
  }
};
exports.validateWildcardInput = validateWildcardInput;
const hasSimpleExecutableName = ({
  os,
  type,
  value
}) => {
  const separator = os === OperatingSystem.WINDOWS ? '\\' : '/';
  const lastString = value.split(separator).pop();
  if (!lastString) {
    return;
  }
  if (type === 'wildcard') {
    return (lastString.split('*').length || lastString.split('?').length) === 1;
  }
  return true;
};
exports.hasSimpleExecutableName = hasSimpleExecutableName;
const isPathValid = ({
  os,
  field,
  type,
  value
}) => {
  const pathFields = ['process.executable.caseless', 'file.path', 'file.path.text'];
  if (pathFields.includes(field)) {
    if (type === 'wildcard') {
      return os === OperatingSystem.WINDOWS ? isWindowsWildcardPathValid(value) : isLinuxMacWildcardPathValid(value);
    }
    return doesPathMatchRegex({
      value,
      os
    });
  }
  return true;
};
exports.isPathValid = isPathValid;
const doesPathMatchRegex = ({
  os,
  value
}) => {
  if (os === OperatingSystem.WINDOWS) {
    const filePathRegex = /^[a-z]:(?:|\\\\[^<>:"'/\\|?*]+\\[^<>:"'/\\|?*]+|%\w+%|)[\\](?:[^<>:"'/\\|?*]+[\\/])*([^<>:"'/\\|?*])+$/i;
    return filePathRegex.test(value);
  }
  return /^(\/|(\/[\w\-]+)+|\/[\w\-]+\.[\w]+|(\/[\w-]+)+\/[\w\-]+\.[\w]+)$/i.test(value);
};
const isWindowsWildcardPathValid = path => {
  const firstCharacter = path[0];
  const lastCharacter = path.slice(-1);
  const trimmedValue = path.trim();
  const hasSlash = /\//.test(trimmedValue);
  if (path.length === 0) {
    return false;
  } else if (hasSlash || trimmedValue.length !== path.length || firstCharacter === '^' || lastCharacter === '\\' || !hasWildcard({
    path,
    isWindowsPath: true
  })) {
    return false;
  } else {
    return true;
  }
};
const isLinuxMacWildcardPathValid = path => {
  const firstCharacter = path[0];
  const lastCharacter = path.slice(-1);
  const trimmedValue = path.trim();
  if (path.length === 0) {
    return false;
  } else if (trimmedValue.length !== path.length || firstCharacter !== '/' || lastCharacter === '/' || path.length > 1024 === true || path.includes('//') === true || !hasWildcard({
    path,
    isWindowsPath: false
  })) {
    return false;
  } else {
    return true;
  }
};
const hasWildcard = ({
  path,
  isWindowsPath
}) => {
  for (const pathComponent of path.split(isWindowsPath ? '\\' : '/')) {
    if (/[\*|\?]+/.test(pathComponent) === true) {
      return true;
    }
  }
  return false;
};