"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VERSION = exports.SecurityUsageReportingTask = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/task-manager-plugin/server");
var _services = require("../common/services");
var _task_state = require("./task_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SCOPE = ['serverlessSecurity'];
const TIMEOUT = '1m';
const VERSION = '1.0.0';
exports.VERSION = VERSION;
class SecurityUsageReportingTask {
  constructor(setupContract) {
    (0, _defineProperty2.default)(this, "wasStarted", false);
    (0, _defineProperty2.default)(this, "cloudSetup", void 0);
    (0, _defineProperty2.default)(this, "taskType", void 0);
    (0, _defineProperty2.default)(this, "version", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "abortController", new AbortController());
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "start", async ({
      taskManager,
      interval
    }) => {
      if (!taskManager) {
        return;
      }
      this.wasStarted = true;
      try {
        await taskManager.ensureScheduled({
          id: this.taskId,
          taskType: this.taskType,
          scope: SCOPE,
          schedule: {
            interval
          },
          state: _task_state.emptyState,
          params: {
            version: this.version
          }
        });
      } catch (e) {
        this.logger.debug(`Error scheduling task, received ${e.message}`);
      }
    });
    (0, _defineProperty2.default)(this, "runTask", async (taskInstance, core, meteringCallback, lookBackLimitMinutes) => {
      // if task was not `.start()`'d yet, then exit
      if (!this.wasStarted) {
        this.logger.debug('[runTask()] Aborted. Task not started yet');
        return;
      }
      // Check that this task is current
      if (taskInstance.id !== this.taskId) {
        // old task, die
        (0, _server.throwUnrecoverableError)(new Error('Outdated task version'));
      }
      const [{
        elasticsearch
      }] = await core.getStartServices();
      const esClient = elasticsearch.client.asInternalUser;
      const lastSuccessfulReport = taskInstance.state.lastSuccessfulReport && new Date(taskInstance.state.lastSuccessfulReport);
      let usageRecords = [];
      // save usage record query time so we can use it to know where
      // the next query range should start
      const meteringCallbackTime = new Date();
      try {
        usageRecords = await meteringCallback({
          esClient,
          cloudSetup: this.cloudSetup,
          logger: this.logger,
          taskId: this.taskId,
          lastSuccessfulReport,
          abortController: this.abortController,
          config: this.config
        });
      } catch (err) {
        this.logger.error(`failed to retrieve usage records: ${err}`);
        return;
      }
      this.logger.debug(`received usage records: ${JSON.stringify(usageRecords)}`);
      let usageReportResponse;
      if (usageRecords.length !== 0) {
        try {
          usageReportResponse = await _services.usageReportingService.reportUsage(usageRecords);
          if (!usageReportResponse.ok) {
            const errorResponse = await usageReportResponse.json();
            this.logger.error(`API error ${usageReportResponse.status}, ${errorResponse}`);
            return;
          }
          this.logger.info(`usage records report was sent successfully: ${usageReportResponse.status}, ${usageReportResponse.statusText}`);
        } catch (err) {
          this.logger.error(`Failed to send usage records report ${err} `);
        }
      }
      const state = {
        lastSuccessfulReport: this.shouldUpdateLastSuccessfulReport(usageRecords, usageReportResponse) ? meteringCallbackTime.toISOString() : this.getFailedLastSuccessfulReportTime(meteringCallbackTime, lastSuccessfulReport, lookBackLimitMinutes).toISOString()
      };
      return {
        state
      };
    });
    const {
      core: _core,
      logFactory,
      config,
      taskManager: _taskManager,
      cloudSetup,
      taskType,
      taskTitle,
      version,
      meteringCallback: _meteringCallback,
      options
    } = setupContract;
    this.cloudSetup = cloudSetup;
    this.taskType = taskType;
    this.version = version;
    this.logger = logFactory.get(this.taskId);
    this.config = config;
    try {
      _taskManager.registerTaskDefinitions({
        [taskType]: {
          title: taskTitle,
          timeout: TIMEOUT,
          stateSchemaByVersion: _task_state.stateSchemaByVersion,
          createTaskRunner: ({
            taskInstance
          }) => {
            return {
              run: async () => {
                return this.runTask(taskInstance, _core, _meteringCallback, options === null || options === void 0 ? void 0 : options.lookBackLimitMinutes);
              },
              cancel: async () => {}
            };
          }
        }
      });
      this.logger.info(`Scheduled task successfully ${taskTitle}`);
    } catch (err) {
      this.logger.error(`Failed to setup task ${taskType}, ${err} `);
    }
  }
  getFailedLastSuccessfulReportTime(meteringCallbackTime, lastSuccessfulReport, lookBackLimitMinutes) {
    const nextLastSuccessfulReport = lastSuccessfulReport || meteringCallbackTime;
    if (!lookBackLimitMinutes) {
      return nextLastSuccessfulReport;
    }
    const lookBackLimitTime = new Date(meteringCallbackTime.setMinutes(-lookBackLimitMinutes));
    if (nextLastSuccessfulReport > lookBackLimitTime) {
      return nextLastSuccessfulReport;
    }
    this.logger.error(`lastSuccessfulReport time of ${nextLastSuccessfulReport.toISOString()} is past the limit of ${lookBackLimitMinutes} minutes, adjusting lastSuccessfulReport to ${lookBackLimitTime.toISOString()}`);
    return lookBackLimitTime;
  }
  shouldUpdateLastSuccessfulReport(usageRecords, usageReportResponse) {
    return !usageRecords.length || (usageReportResponse === null || usageReportResponse === void 0 ? void 0 : usageReportResponse.status) === 201;
  }
  get taskId() {
    return `${this.taskType}:${this.version}`;
  }
}
exports.SecurityUsageReportingTask = SecurityUsageReportingTask;