"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerResponseActionRoutes = registerResponseActionRoutes;
var _custom_http_request_error = require("../../../utils/custom_http_request_error");
var _clients = require("../../services/actions/clients");
var _endpoint = require("../../../../common/api/endpoint");
var _constants = require("../../../../common/endpoint/constants");
var _with_endpoint_authz = require("../with_endpoint_authz");
var _error_handler = require("../error_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerResponseActionRoutes(router, endpointContext) {
  const logger = endpointContext.logFactory.get('hostIsolation');

  /**
   * @deprecated use ISOLATE_HOST_ROUTE_V2 instead
   */
  router.versioned.post({
    access: 'public',
    path: _constants.ISOLATE_HOST_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.IsolateRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canIsolateHost']
  }, logger, redirectHandler(_constants.ISOLATE_HOST_ROUTE_V2)));

  /**
   * @deprecated use RELEASE_HOST_ROUTE instead
   */
  router.versioned.post({
    access: 'public',
    path: _constants.UNISOLATE_HOST_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.UnisolateRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canUnIsolateHost']
  }, logger, redirectHandler(_constants.UNISOLATE_HOST_ROUTE_V2)));
  router.versioned.post({
    access: 'public',
    path: _constants.ISOLATE_HOST_ROUTE_V2,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.IsolateRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canIsolateHost']
  }, logger, responseActionRequestHandler(endpointContext, 'isolate')));
  router.versioned.post({
    access: 'public',
    path: _constants.UNISOLATE_HOST_ROUTE_V2,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.UnisolateRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canUnIsolateHost']
  }, logger, responseActionRequestHandler(endpointContext, 'unisolate')));
  router.versioned.post({
    access: 'public',
    path: _constants.KILL_PROCESS_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.KillProcessRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canKillProcess']
  }, logger, responseActionRequestHandler(endpointContext, 'kill-process')));
  router.versioned.post({
    access: 'public',
    path: _constants.SUSPEND_PROCESS_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.SuspendProcessRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canSuspendProcess']
  }, logger, responseActionRequestHandler(endpointContext, 'suspend-process')));
  router.versioned.post({
    access: 'public',
    path: _constants.GET_PROCESSES_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.GetProcessesRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canGetRunningProcesses']
  }, logger, responseActionRequestHandler(endpointContext, 'running-processes')));
  router.versioned.post({
    access: 'public',
    path: _constants.GET_FILE_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.EndpointActionGetFileSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteFileOperations']
  }, logger, responseActionRequestHandler(endpointContext, 'get-file')));
  router.versioned.post({
    access: 'public',
    path: _constants.EXECUTE_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.ExecuteActionRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteExecuteOperations']
  }, logger, responseActionRequestHandler(endpointContext, 'execute')));
  router.versioned.post({
    access: 'public',
    path: _constants.UPLOAD_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution'],
      body: {
        accepts: ['multipart/form-data'],
        output: 'stream',
        maxBytes: endpointContext.serverConfig.maxUploadResponseActionFileBytes
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.UploadActionRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteFileOperations']
  }, logger, responseActionRequestHandler(endpointContext, 'upload')));
}
function responseActionRequestHandler(endpointContext, command) {
  const logger = endpointContext.logFactory.get('responseActionsHandler');
  return async (context, req, res) => {
    var _endpointContext$serv, _user$username;
    const user = (_endpointContext$serv = endpointContext.service.security) === null || _endpointContext$serv === void 0 ? void 0 : _endpointContext$serv.authc.getCurrentUser(req);
    const esClient = (await context.core).elasticsearch.client.asInternalUser;
    const casesClient = await endpointContext.service.getCasesClient(req);
    const actionsClient = new _clients.EndpointActionsClient({
      esClient,
      casesClient,
      endpointContext,
      username: (_user$username = user === null || user === void 0 ? void 0 : user.username) !== null && _user$username !== void 0 ? _user$username : 'unknown'
    });
    try {
      let action;
      switch (command) {
        case 'isolate':
          action = await actionsClient.isolate(req.body);
          break;
        case 'unisolate':
          action = await actionsClient.release(req.body);
          break;
        case 'running-processes':
          action = await actionsClient.runningProcesses(req.body);
          break;
        case 'execute':
          action = await actionsClient.execute(req.body);
          break;
        case 'suspend-process':
          action = await actionsClient.suspendProcess(req.body);
          break;
        case 'kill-process':
          action = await actionsClient.killProcess(req.body);
          break;
        case 'get-file':
          action = await actionsClient.getFile(req.body);
          break;
        case 'upload':
          action = await actionsClient.upload(req.body);
          break;
        default:
          throw new _custom_http_request_error.CustomHttpRequestError(`No handler found for response action command: [${command}]`, 501);
      }
      const {
        action: actionId,
        ...data
      } = action;
      return res.ok({
        body: {
          action: actionId,
          data
        }
      });
    } catch (err) {
      return (0, _error_handler.errorHandler)(logger, res, err);
    }
  };
}
function redirectHandler(location) {
  return async (context, _req, res) => {
    const basePath = (await context.securitySolution).getServerBasePath();
    return res.custom({
      statusCode: 308,
      headers: {
        location: `${basePath}${location}`
      }
    });
  };
}