"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useGetEndpointsList = void 0;
var _reactQuery = require("@tanstack/react-query");
var _kibana = require("../../../common/lib/kibana");
var _constants = require("../../../../common/endpoint/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const PAGING_PARAMS = Object.freeze({
  default: 50,
  all: 10000
});
/**
 * Get info for a security solution endpoint host using hostnames via kuery param
 * page and pageSize are fixed for showing 50 hosts at most on the 1st page
 * @param query
 * @param options
 */
const useGetEndpointsList = ({
  searchString,
  selectedAgentIds,
  options = {}
}) => {
  const http = (0, _kibana.useHttp)();
  const kuery = `united.endpoint.host.hostname:${searchString.length ? `*${searchString}` : ''}*`;
  let agentIdsKuery = [];
  if (selectedAgentIds) {
    agentIdsKuery = selectedAgentIds.map(id => `united.endpoint.agent.id:"${id}"`);
  }
  return (0, _reactQuery.useQuery)({
    queryKey: ['get-endpoints-list', kuery],
    ...options,
    queryFn: async () => {
      const metadataListResponse = await http.get(_constants.HOST_METADATA_LIST_ROUTE, {
        version: '2023-10-31',
        query: {
          page: 0,
          pageSize:
          // if the user has selected agents then search the whole index.
          // as selected host could be somewhere after the 50 that are shown
          // otherwise, limit the search to 50 hosts
          selectedAgentIds && selectedAgentIds.length > 0 ? PAGING_PARAMS.all : PAGING_PARAMS.default,
          kuery: [...agentIdsKuery, kuery].join(' or ')
        }
      });

      // pick out the selected agents and push them to the top of the list
      const augmentedDataBasedOnSelectedAgents = metadataListResponse.data.reduce((acc, list) => {
        const item = {
          id: list.metadata.agent.id,
          name: list.metadata.host.hostname
        };
        if (selectedAgentIds !== null && selectedAgentIds !== void 0 && selectedAgentIds.includes(list.metadata.agent.id)) {
          acc.selected.push({
            ...item,
            selected: true
          });
        } else {
          acc.rest.push({
            ...item,
            selected: false
          });
        }
        return acc;
      }, {
        selected: [],
        rest: []
      });
      let selectedAgentIdsCount = 0;
      if (selectedAgentIds) {
        selectedAgentIdsCount = selectedAgentIds.length;
      }

      // return 50 items max including the selected items
      // unless all 50 items are selected then increase the list length by 10
      return [...augmentedDataBasedOnSelectedAgents.selected, ...augmentedDataBasedOnSelectedAgents.rest].slice(0, selectedAgentIdsCount >= PAGING_PARAMS.default ? selectedAgentIdsCount + 10 : PAGING_PARAMS.default);
    }
  });
};
exports.useGetEndpointsList = useGetEndpointsList;