"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEventCategoriesFromData = getEventCategoriesFromData;
exports.getEventFieldsToDisplay = getEventFieldsToDisplay;
exports.getSummaryRows = void 0;
var _fp = require("lodash/fp");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _securitysolutionEcs = require("@kbn/securitysolution-ecs");
var i18n = _interopRequireWildcard(require("./translations"));
var _translations2 = require("../../../detections/components/alerts_table/translations");
var _field_names = require("../../../../common/field_maps/field_names");
var _constants = require("../../../timelines/components/timeline/body/renderers/constants");
var _helpers = require("./helpers");
var _endpoint_alert_check = require("../../utils/endpoint_alert_check");
var _sentinelone_alert_check = require("../../utils/sentinelone_alert_check");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const THRESHOLD_TERMS_FIELD = `${_field_names.ALERT_THRESHOLD_RESULT}.terms.field`;
const THRESHOLD_TERMS_VALUE = `${_field_names.ALERT_THRESHOLD_RESULT}.terms.value`;
const THRESHOLD_CARDINALITY_FIELD = `${_field_names.ALERT_THRESHOLD_RESULT}.cardinality.field`;
const THRESHOLD_CARDINALITY_VALUE = `${_field_names.ALERT_THRESHOLD_RESULT}.cardinality.value`;
const THRESHOLD_COUNT = `${_field_names.ALERT_THRESHOLD_RESULT}.count`;

/** Always show these fields */
const alwaysDisplayedFields = [{
  id: 'host.name'
},
// ENDPOINT-related field //
{
  id: 'agent.id',
  overrideField: _constants.AGENT_STATUS_FIELD_NAME,
  label: i18n.AGENT_STATUS
}, {
  id: _sentinelone_alert_check.SENTINEL_ONE_AGENT_ID_FIELD,
  overrideField: _constants.AGENT_STATUS_FIELD_NAME,
  label: i18n.AGENT_STATUS
},
// ** //
{
  id: 'user.name'
}, {
  id: 'rule.name'
}, {
  id: 'cloud.provider'
}, {
  id: 'cloud.region'
}, {
  id: 'cloud.provider'
}, {
  id: 'cloud.region'
}, {
  id: 'orchestrator.cluster.id'
}, {
  id: 'orchestrator.cluster.name'
}, {
  id: 'container.image.name'
}, {
  id: 'container.image.tag'
}, {
  id: 'orchestrator.namespace'
}, {
  id: 'orchestrator.resource.parent.type'
}, {
  id: 'orchestrator.resource.type'
}, {
  id: 'process.executable'
}, {
  id: 'file.path'
}, {
  id: _ruleDataUtils.ALERT_RULE_TYPE,
  label: i18n.RULE_TYPE
}];

/**
 * Get a list of fields to display based on the event's category
 */
function getFieldsByCategory({
  primaryEventCategory,
  allEventCategories
}) {
  switch (primaryEventCategory) {
    case _securitysolutionEcs.EventCategory.PROCESS:
      return [{
        id: 'process.name'
      }, {
        id: 'process.parent.name'
      }, {
        id: 'process.args'
      }];
    case _securitysolutionEcs.EventCategory.FILE:
      return [{
        id: 'file.name'
      }, {
        id: 'file.hash.sha256'
      }, {
        id: 'file.directory'
      }, {
        id: 'process.name'
      }];
    case _securitysolutionEcs.EventCategory.NETWORK:
      return [{
        id: 'destination.address'
      }, {
        id: 'destination.port'
      }, {
        id: 'source.address'
      }, {
        id: 'source.port'
      }, {
        id: 'dns.question.name'
      }, {
        id: 'process.name'
      }];
    case _securitysolutionEcs.EventCategory.REGISTRY:
      return [{
        id: 'registry.key'
      }, {
        id: 'registry.value'
      }, {
        id: 'process.name'
      }];
    case _securitysolutionEcs.EventCategory.MALWARE:
      // The details for malware events can be found in the file fields
      return getFieldsByCategory({
        primaryEventCategory: _securitysolutionEcs.EventCategory.FILE,
        allEventCategories
      });
    default:
      let fields = [];

      // If no primary category matches or hasn't been defined on purpose (e.g. in order to follow the source event)
      // resolve more fields based on the other event categories.
      if (allEventCategories !== null && allEventCategories !== void 0 && allEventCategories.includes(_securitysolutionEcs.EventCategory.FILE)) {
        fields = fields.concat(getFieldsByCategory({
          primaryEventCategory: _securitysolutionEcs.EventCategory.FILE
        }));
      }
      if (allEventCategories !== null && allEventCategories !== void 0 && allEventCategories.includes(_securitysolutionEcs.EventCategory.PROCESS)) {
        fields = fields.concat(getFieldsByCategory({
          primaryEventCategory: _securitysolutionEcs.EventCategory.PROCESS
        }));
      }
      return fields;
  }
}

/**
 * Gets the fields to display based on the event's code.
 * Contains some enhancements to resolve more fields based on the event's categories.
 * @param eventCode The event's code
 * @param eventCategories The events categories
 * @returns A list of fields to include
 */
function getFieldsByEventCode(eventCode, eventCategories) {
  switch (eventCode) {
    case _securitysolutionEcs.EventCode.BEHAVIOR:
      return [{
        id: 'rule.description',
        label: _translations2.ALERTS_HEADERS_RULE_DESCRIPTION
      },
      // Resolve more fields based on the source event
      ...getFieldsByCategory({
        ...eventCategories,
        primaryEventCategory: undefined
      })];
    case _securitysolutionEcs.EventCode.SHELLCODE_THREAD:
      return [{
        id: 'Target.process.executable'
      }, {
        id: 'Memory_protection.unique_key_v1'
      }];
    case _securitysolutionEcs.EventCode.RANSOMWARE:
      return [{
        id: 'Ransomware.feature'
      }, {
        id: 'process.hash.sha256'
      }, ...getFieldsByCategory({
        ...eventCategories,
        primaryEventCategory: undefined
      })];
    case _securitysolutionEcs.EventCode.MEMORY_SIGNATURE:
      // Resolve more fields based on the source event
      return getFieldsByCategory({
        ...eventCategories,
        primaryEventCategory: undefined
      });
    case _securitysolutionEcs.EventCode.MALICIOUS_FILE:
      return [{
        id: 'file.Ext.quarantine_path',
        overrideField: _constants.QUARANTINED_PATH_FIELD_NAME,
        label: i18n.QUARANTINED_FILE_PATH
      }];
    default:
      return [];
  }
}

/**
 * Returns a list of fields based on the event's rule type
 */
function getFieldsByRuleType(ruleType) {
  switch (ruleType) {
    case 'threshold':
      return [{
        id: THRESHOLD_COUNT,
        label: _translations2.ALERTS_HEADERS_THRESHOLD_COUNT
      }, {
        id: THRESHOLD_TERMS_FIELD,
        overrideField: THRESHOLD_TERMS_VALUE,
        label: _translations2.ALERTS_HEADERS_THRESHOLD_TERMS
      }, {
        id: THRESHOLD_CARDINALITY_FIELD,
        label: _translations2.ALERTS_HEADERS_THRESHOLD_CARDINALITY
      }];
    case 'machine_learning':
      return [{
        id: `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.machine_learning_job_id`,
        legacyId: 'signal.rule.machine_learning_job_id'
      }, {
        id: `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.anomaly_threshold`,
        legacyId: 'signal.rule.anomaly_threshold'
      }];
    case 'threat_match':
      return [{
        id: `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.threat_index`,
        legacyId: 'signal.rule.threat_index'
      }, {
        id: `${_ruleDataUtils.ALERT_RULE_PARAMETERS}.threat_query`,
        legacyId: 'signal.rule.threat_query'
      }];
    case 'new_terms':
      return [{
        id: _field_names.ALERT_NEW_TERMS_FIELDS,
        label: _translations2.ALERTS_HEADERS_NEW_TERMS_FIELDS
      }, {
        id: _field_names.ALERT_NEW_TERMS,
        label: _translations2.ALERTS_HEADERS_NEW_TERMS
      }];
    default:
      return [];
  }
}

/**
 * Gets the fields to display based on custom rules and configuration
 * @param customs The list of custom-defined fields to display
 * @returns The list of custom-defined fields to display
 */
function getHighlightedFieldsOverride(customs) {
  return customs.map(field => ({
    id: field
  }));
}

/**
  This function is exported because it is used in the Exception Component to
  populate the conditions with the Highlighted Fields. Additionally, the new
  Alert Summary Flyout also requires access to these fields.
  As the Alert Summary components will undergo changes soon we will go with
  exporting the function only for now.
 */
/**
 * Assembles a list of fields to display based on the event
 */
function getEventFieldsToDisplay({
  eventCategories,
  eventCode,
  eventRuleType,
  highlightedFieldsOverride
}) {
  const fields = [...getHighlightedFieldsOverride(highlightedFieldsOverride), ...alwaysDisplayedFields, ...getFieldsByCategory(eventCategories), ...getFieldsByEventCode(eventCode, eventCategories), ...getFieldsByRuleType(eventRuleType)];

  // Filter all fields by their id to make sure there are no duplicates
  return (0, _fp.uniqBy)('id', fields);
}
/**
 * Extract the event's categories
 * @param data The event details
 * @returns The event's primary category and all other categories in case there is more than one
 */
function getEventCategoriesFromData(data) {
  const eventCategoryField = (0, _fp.find)({
    category: 'event',
    field: 'event.category'
  }, data);
  let primaryEventCategory;
  let allEventCategories;
  if (Array.isArray(eventCategoryField === null || eventCategoryField === void 0 ? void 0 : eventCategoryField.originalValue)) {
    primaryEventCategory = eventCategoryField === null || eventCategoryField === void 0 ? void 0 : eventCategoryField.originalValue[0];
    allEventCategories = eventCategoryField === null || eventCategoryField === void 0 ? void 0 : eventCategoryField.originalValue;
  } else {
    primaryEventCategory = eventCategoryField === null || eventCategoryField === void 0 ? void 0 : eventCategoryField.originalValue;
    if (primaryEventCategory) {
      allEventCategories = [primaryEventCategory];
    }
  }
  return {
    primaryEventCategory,
    allEventCategories
  };
}
const getSummaryRows = ({
  data,
  browserFields,
  scopeId,
  eventId,
  isDraggable = false,
  isReadOnly = false,
  investigationFields,
  sentinelOneManualHostActionsEnabled
}) => {
  var _eventCodeField$origi, _eventRuleTypeField$o;
  const eventCategories = getEventCategoriesFromData(data);
  const eventCodeField = (0, _fp.find)({
    category: 'event',
    field: 'event.code'
  }, data);
  const eventCode = Array.isArray(eventCodeField === null || eventCodeField === void 0 ? void 0 : eventCodeField.originalValue) ? eventCodeField === null || eventCodeField === void 0 ? void 0 : (_eventCodeField$origi = eventCodeField.originalValue) === null || _eventCodeField$origi === void 0 ? void 0 : _eventCodeField$origi[0] : eventCodeField === null || eventCodeField === void 0 ? void 0 : eventCodeField.originalValue;
  const eventRuleTypeField = (0, _fp.find)({
    category: 'kibana',
    field: _ruleDataUtils.ALERT_RULE_TYPE
  }, data);
  const eventRuleType = Array.isArray(eventRuleTypeField === null || eventRuleTypeField === void 0 ? void 0 : eventRuleTypeField.originalValue) ? eventRuleTypeField === null || eventRuleTypeField === void 0 ? void 0 : (_eventRuleTypeField$o = eventRuleTypeField.originalValue) === null || _eventRuleTypeField$o === void 0 ? void 0 : _eventRuleTypeField$o[0] : eventRuleTypeField === null || eventRuleTypeField === void 0 ? void 0 : eventRuleTypeField.originalValue;
  const tableFields = getEventFieldsToDisplay({
    eventCategories,
    eventCode,
    eventRuleType,
    highlightedFieldsOverride: investigationFields !== null && investigationFields !== void 0 ? investigationFields : []
  });
  return data != null ? tableFields.reduce((acc, field) => {
    var _field$label;
    const item = data.find(d => d.field === field.id || field.legacyId && d.field === field.legacyId);
    if (!item || (0, _fp.isEmpty)(item.values)) {
      return acc;
    }

    // If we found the data by its legacy id we swap the ids to display the correct one
    if (item.field === field.legacyId) {
      field.id = field.legacyId;
    }
    const linkValueField = field.linkField != null && data.find(d => d.field === field.linkField);
    const description = {
      ...(0, _helpers.getEnrichedFieldInfo)({
        item,
        linkValueField: linkValueField || undefined,
        contextId: scopeId,
        scopeId,
        browserFields,
        eventId,
        field
      }),
      isDraggable,
      isReadOnly
    };
    if (field.id === 'agent.id' && !(0, _endpoint_alert_check.isAlertFromEndpointEvent)({
      data
    })) {
      return acc;
    }
    if (field.id === _sentinelone_alert_check.SENTINEL_ONE_AGENT_ID_FIELD && sentinelOneManualHostActionsEnabled && !(0, _sentinelone_alert_check.isAlertFromSentinelOneEvent)({
      data
    })) {
      return acc;
    }
    if (field.id === THRESHOLD_TERMS_FIELD) {
      const enrichedInfo = enrichThresholdTerms(item, data, description);
      if (enrichedInfo) {
        return [...acc, ...enrichedInfo];
      } else {
        return acc;
      }
    }
    if (field.id === THRESHOLD_CARDINALITY_FIELD) {
      const enrichedInfo = enrichThresholdCardinality(item, data, description);
      if (enrichedInfo) {
        return [...acc, enrichedInfo];
      } else {
        return acc;
      }
    }
    return [...acc, {
      title: (_field$label = field.label) !== null && _field$label !== void 0 ? _field$label : field.id,
      description
    }];
  }, []) : [];
};

/**
 * Enriches the summary data for threshold terms.
 * For any given threshold term, it generates a row with the term's name and the associated value.
 */
exports.getSummaryRows = getSummaryRows;
function enrichThresholdTerms({
  values: termsFieldArr
}, data, description) {
  const termsValueItem = data.find(d => d.field === THRESHOLD_TERMS_VALUE);
  const termsValueArray = termsValueItem && termsValueItem.values;

  // Make sure both `fields` and `values` are an array and that they have the same length
  if (Array.isArray(termsFieldArr) && termsFieldArr.length > 0 && Array.isArray(termsValueArray) && termsFieldArr.length === termsValueArray.length) {
    return termsFieldArr.map((field, index) => ({
      title: field,
      description: {
        ...description,
        values: [termsValueArray[index]]
      }
    })).filter(entry => !alwaysDisplayedFields.map(alwaysThereEntry => alwaysThereEntry.id).includes(entry.title));
  }
}

/**
 * Enriches the summary data for threshold cardinality.
 * Reads out the cardinality field and the value and interpolates them into a combined string value.
 */
function enrichThresholdCardinality({
  values: cardinalityFieldArr
}, data, description) {
  const cardinalityValueItem = data.find(d => d.field === THRESHOLD_CARDINALITY_VALUE);
  const cardinalityValueArray = cardinalityValueItem && cardinalityValueItem.values;

  // Only return a summary row if we actually have the correct field and value
  if (Array.isArray(cardinalityFieldArr) && cardinalityFieldArr.length === 1 && Array.isArray(cardinalityValueArray) && cardinalityFieldArr.length === cardinalityValueArray.length) {
    return {
      title: _translations2.ALERTS_HEADERS_THRESHOLD_CARDINALITY,
      description: {
        ...description,
        values: [`count(${cardinalityFieldArr[0]}) >= ${cardinalityValueArray[0]}`]
      }
    };
  }
}