"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchConnectors = exports.fetchConnectorByIndexName = exports.fetchConnectorById = void 0;
var _identify_exceptions = require("../utils/identify_exceptions");
var _ = require("..");
var _is_not_nullish = require("../utils/is_not_nullish");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const fetchConnectorById = async (client, connectorId) => {
  try {
    const connectorResult = await client.get({
      id: connectorId,
      index: _.CONNECTORS_INDEX
    });
    return connectorResult._source ? {
      primaryTerm: connectorResult._primary_term,
      seqNo: connectorResult._seq_no,
      value: {
        ...connectorResult._source,
        id: connectorResult._id
      }
    } : undefined;
  } catch (error) {
    if ((0, _identify_exceptions.isIndexNotFoundException)(error)) {
      return undefined;
    }
    throw error;
  }
};
exports.fetchConnectorById = fetchConnectorById;
const fetchConnectorByIndexName = async (client, indexName) => {
  try {
    var _connectorResult$hits, _await$fetchConnector;
    const connectorResult = await client.search({
      index: _.CONNECTORS_INDEX,
      query: {
        term: {
          index_name: indexName
        }
      }
    });
    // Because we cannot guarantee that the index has been refreshed and is giving us the most recent source
    // we need to fetch the source with a direct get from the index, which will always be up to date
    const result = (_connectorResult$hits = connectorResult.hits.hits[0]) !== null && _connectorResult$hits !== void 0 && _connectorResult$hits._source ? (_await$fetchConnector = await fetchConnectorById(client, connectorResult.hits.hits[0]._id)) === null || _await$fetchConnector === void 0 ? void 0 : _await$fetchConnector.value : undefined;
    return result;
  } catch (error) {
    if ((0, _identify_exceptions.isIndexNotFoundException)(error)) {
      return undefined;
    }
    throw error;
  }
};
exports.fetchConnectorByIndexName = fetchConnectorByIndexName;
const fetchConnectors = async (client, indexNames, fetchOnlyCrawlers, searchQuery) => {
  const q = searchQuery && searchQuery.length > 0 ? searchQuery : undefined;
  const query = q ? {
    bool: {
      should: [{
        wildcard: {
          name: {
            value: `*${q}*`
          }
        }
      }, {
        wildcard: {
          index_name: {
            value: `*${q}*`
          }
        }
      }]
    }
  } : indexNames ? {
    terms: {
      index_name: indexNames
    }
  } : {
    match_all: {}
  };
  try {
    let hits = [];
    let accumulator = [];
    do {
      const connectorResult = await client.search({
        from: accumulator.length,
        index: _.CONNECTORS_INDEX,
        query,
        size: 1000
      });
      hits = connectorResult.hits.hits;
      accumulator = accumulator.concat(hits);
    } while (hits.length >= 1000);
    const result = accumulator.map(({
      _source,
      _id
    }) => _source ? {
      ..._source,
      id: _id
    } : undefined).filter(_is_not_nullish.isNotNullish);
    if (fetchOnlyCrawlers !== undefined) {
      return result.filter(hit => {
        return !fetchOnlyCrawlers ? hit.service_type !== _.CRAWLER_SERVICE_TYPE : hit.service_type === _.CRAWLER_SERVICE_TYPE;
      });
    }
    return result;
  } catch (error) {
    if ((0, _identify_exceptions.isIndexNotFoundException)(error)) {
      return [];
    }
    throw error;
  }
};
exports.fetchConnectors = fetchConnectors;