"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.STEP_TERMS = exports.STEP_REVIEW = exports.STEP_METRICS = exports.STEP_LOGISTICS = exports.STEP_HISTOGRAM = exports.STEP_DATE_HISTOGRAM = void 0;
exports.getAffectedStepsFields = getAffectedStepsFields;
exports.hasErrors = hasErrors;
exports.stepIds = exports.stepIdToStepConfigMap = void 0;
var _lodash = require("lodash");
var _validate_id = require("./validate_id");
var _validate_index_pattern = require("./validate_index_pattern");
var _validate_rollup_index = require("./validate_rollup_index");
var _validate_rollup_cron = require("./validate_rollup_cron");
var _validate_rollup_page_size = require("./validate_rollup_page_size");
var _validate_rollup_delay = require("./validate_rollup_delay");
var _validate_date_histogram_field = require("./validate_date_histogram_field");
var _validate_date_histogram_interval = require("./validate_date_histogram_interval");
var _validate_histogram_interval = require("./validate_histogram_interval");
var _validate_metrics = require("./validate_metrics");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const STEP_LOGISTICS = 'STEP_LOGISTICS';
exports.STEP_LOGISTICS = STEP_LOGISTICS;
const STEP_DATE_HISTOGRAM = 'STEP_DATE_HISTOGRAM';
exports.STEP_DATE_HISTOGRAM = STEP_DATE_HISTOGRAM;
const STEP_TERMS = 'STEP_TERMS';
exports.STEP_TERMS = STEP_TERMS;
const STEP_HISTOGRAM = 'STEP_HISTOGRAM';
exports.STEP_HISTOGRAM = STEP_HISTOGRAM;
const STEP_METRICS = 'STEP_METRICS';
exports.STEP_METRICS = STEP_METRICS;
const STEP_REVIEW = 'STEP_REVIEW';
exports.STEP_REVIEW = STEP_REVIEW;
const stepIds = [STEP_LOGISTICS, STEP_DATE_HISTOGRAM, STEP_TERMS, STEP_HISTOGRAM, STEP_METRICS, STEP_REVIEW];

/**
 * Map a specific wizard step to two functions:
 *  1. getDefaultFields: (overrides) => object
 *  2. fieldValidations
 *
 * See rollup/public/crud_app/services/jobs.js for more information on override's shape
 */
exports.stepIds = stepIds;
const stepIdToStepConfigMap = {
  [STEP_LOGISTICS]: {
    getDefaultFields: (overrides = {}) => {
      // We don't display the simple editor if there are overrides for the rollup's cron
      const isAdvancedCronVisible = !!overrides.rollupCron;

      // The best page size boils down to how much memory the user has, e.g. how many buckets should
      // be accumulated at one time. 1000 is probably a safe size without being too small.
      const rollupPageSize = (0, _lodash.get)(overrides, ['json', 'config', 'page_size'], 1000);
      const clonedRollupId = overrides.id || undefined;
      const id = overrides.id ? `${overrides.id}-copy` : '';
      const defaults = {
        indexPattern: '',
        rollupIndex: '',
        // Every week on Saturday, at 00:00:00
        rollupCron: '0 0 0 ? * 7',
        simpleRollupCron: '0 0 0 ? * 7',
        rollupPageSize,
        // Though the API doesn't require a delay, in many real-world cases, servers will go down for
        // a few hours as they're being restarted. A delay of 1d would allow them that period to reboot
        // and the "expense" is pretty negligible in most cases: 1 day of extra non-rolled-up data.
        rollupDelay: '1d',
        cronFrequency: 'WEEK',
        fieldToPreferredValueMap: {}
      };
      return {
        ...defaults,
        ...(0, _lodash.pick)(overrides, Object.keys(defaults)),
        id,
        isAdvancedCronVisible,
        rollupPageSize,
        clonedRollupId
      };
    },
    fieldsValidator: fields => {
      const {
        id,
        indexPattern,
        rollupIndex,
        rollupCron,
        rollupPageSize,
        rollupDelay,
        clonedRollupId
      } = fields;
      return {
        id: (0, _validate_id.validateId)(id, clonedRollupId),
        indexPattern: (0, _validate_index_pattern.validateIndexPattern)(indexPattern, rollupIndex),
        rollupIndex: (0, _validate_rollup_index.validateRollupIndex)(rollupIndex, indexPattern),
        rollupCron: (0, _validate_rollup_cron.validateRollupCron)(rollupCron),
        rollupPageSize: (0, _validate_rollup_page_size.validateRollupPageSize)(rollupPageSize),
        rollupDelay: (0, _validate_rollup_delay.validateRollupDelay)(rollupDelay)
      };
    }
  },
  [STEP_DATE_HISTOGRAM]: {
    getDefaultFields: (overrides = {}) => {
      const defaults = {
        dateHistogramField: null,
        dateHistogramInterval: null,
        dateHistogramTimeZone: 'UTC'
      };
      return {
        ...defaults,
        ...(0, _lodash.pick)(overrides, Object.keys(defaults))
      };
    },
    fieldsValidator: fields => {
      const {
        dateHistogramField,
        dateHistogramInterval
      } = fields;
      return {
        dateHistogramField: (0, _validate_date_histogram_field.validateDateHistogramField)(dateHistogramField),
        dateHistogramInterval: (0, _validate_date_histogram_interval.validateDateHistogramInterval)(dateHistogramInterval)
      };
    }
  },
  [STEP_TERMS]: {
    getDefaultFields: (overrides = {}) => {
      return {
        terms: [],
        ...(0, _lodash.pick)(overrides, ['terms'])
      };
    }
  },
  [STEP_HISTOGRAM]: {
    getDefaultFields: overrides => {
      return {
        histogram: [],
        histogramInterval: undefined,
        ...(0, _lodash.pick)(overrides, ['histogram', 'histogramInterval'])
      };
    },
    fieldsValidator: fields => {
      const {
        histogram,
        histogramInterval
      } = fields;
      return {
        histogramInterval: (0, _validate_histogram_interval.validateHistogramInterval)(histogram, histogramInterval)
      };
    }
  },
  [STEP_METRICS]: {
    getDefaultFields: (overrides = {}) => {
      return {
        metrics: [],
        ...(0, _lodash.pick)(overrides, ['metrics'])
      };
    },
    fieldsValidator: fields => {
      const {
        metrics
      } = fields;
      return {
        metrics: (0, _validate_metrics.validateMetrics)(metrics)
      };
    }
  },
  [STEP_REVIEW]: {
    getDefaultFields: () => ({})
  }
};
exports.stepIdToStepConfigMap = stepIdToStepConfigMap;
function getAffectedStepsFields(fields, stepsFields) {
  const {
    indexPattern
  } = fields;
  const affectedStepsFields = (0, _lodash.cloneDeep)(stepsFields);

  // A new index pattern means we have to clear all of the fields which depend upon it.
  if (indexPattern) {
    affectedStepsFields[STEP_DATE_HISTOGRAM].dateHistogramField = undefined;
    affectedStepsFields[STEP_TERMS].terms = [];
    affectedStepsFields[STEP_HISTOGRAM].histogram = [];
    affectedStepsFields[STEP_METRICS].metrics = [];
  }
  return affectedStepsFields;
}
function hasErrors(fieldErrors) {
  const errorValues = Object.values(fieldErrors);
  return errorValues.some(error => error !== undefined);
}