"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.commonJobsRouteHandlerFactory = void 0;
var _configSchema = require("@kbn/config-schema");
var _reportingCommon = require("@kbn/reporting-common");
var _ = require("..");
var _lib = require("../../../lib");
var _generate = require("../generate");
var _job_management_pre_routing = require("./job_management_pre_routing");
var _jobs_query = require("./jobs_query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const validate = {
  params: _configSchema.schema.object({
    docId: _configSchema.schema.string({
      minLength: 3
    })
  })
};
const commonJobsRouteHandlerFactory = reporting => {
  const jobsQuery = (0, _jobs_query.jobsQueryFactory)(reporting);
  const handleDownloadReport = ({
    path,
    user,
    context,
    req,
    res
  }) => {
    const counters = (0, _.getCounters)(req.route.method, path, reporting.getUsageCounter());

    // ensure the async dependencies are loaded
    if (!context.reporting) {
      return (0, _generate.handleUnavailable)(res);
    }
    const {
      docId
    } = req.params;
    return (0, _job_management_pre_routing.jobManagementPreRouting)(reporting, res, docId, user, counters, async doc => {
      const payload = await jobsQuery.getDocumentPayload(doc);
      const {
        contentType,
        content,
        filename,
        statusCode
      } = payload;
      if (!contentType || !_reportingCommon.ALLOWED_JOB_CONTENT_TYPES.includes(contentType)) {
        return res.badRequest({
          body: `Unsupported content-type of ${contentType} specified by job output`
        });
      }
      const body = typeof content === 'string' ? Buffer.from(content) : content;
      const headers = {
        ...payload.headers,
        'content-type': contentType
      };
      if (filename) {
        return res.file({
          body,
          headers,
          filename
        });
      }
      return res.custom({
        body,
        headers,
        statusCode
      });
    });
  };
  const handleDeleteReport = ({
    path,
    user,
    context,
    req,
    res
  }) => {
    const counters = (0, _.getCounters)(req.route.method, path, reporting.getUsageCounter());

    // ensure the async dependencies are loaded
    if (!context.reporting) {
      return (0, _generate.handleUnavailable)(res);
    }
    const {
      docId
    } = req.params;
    return (0, _job_management_pre_routing.jobManagementPreRouting)(reporting, res, docId, user, counters, async doc => {
      const docIndex = doc.index;
      const stream = await (0, _lib.getContentStream)(reporting, {
        id: docId,
        index: docIndex
      });
      const reportingSetup = reporting.getPluginSetupDeps();
      const logger = reportingSetup.logger.get('delete-report');

      // An "error" event is emitted if an error is
      // passed to the `stream.end` callback from
      // the _final method of the ContentStream.
      // This event must be handled.
      stream.on('error', err => {
        logger.error(err);
      });
      try {
        // Overwriting existing content with an
        // empty buffer to remove all the chunks.
        await new Promise((resolve, reject) => {
          stream.end('', 'utf8', error => {
            if (error) {
              // handle error that could be thrown
              // from the _write method of the ContentStream
              reject(error);
            } else {
              resolve();
            }
          });
        });
        await jobsQuery.delete(docIndex, docId);
        return res.ok({
          body: {
            deleted: true
          }
        });
      } catch (error) {
        logger.error(error);
        return res.customError({
          statusCode: 500
        });
      }
    });
  };
  return {
    validate,
    handleDownloadReport,
    handleDeleteReport
  };
};
exports.commonJobsRouteHandlerFactory = commonJobsRouteHandlerFactory;