"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DifferentialTopNFunctionsGrid = DifferentialTopNFunctionsGrid;
exports.getFrameIdentification = getFrameIdentification;
exports.isComparisonColumn = void 0;
var _eui = require("@elastic/eui");
var _react = require("@emotion/react");
var _i18n = require("@kbn/i18n");
var _profilingUtils = require("@kbn/profiling-utils");
var _lodash = require("lodash");
var _react2 = _interopRequireWildcard(require("react"));
var _use_calculate_impact_estimates = require("../../hooks/use_calculate_impact_estimates");
var _function_row = require("../topn_functions/function_row");
var _utils = require("../topn_functions/utils");
var _get_columns = require("./get_columns");
var _get_compare_frame_action = require("./get_compare_frame_action");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const removeComparisonFromId = id => id.replace('comparison_', '');
const isComparisonColumn = id => id.startsWith('comparison_');
exports.isComparisonColumn = isComparisonColumn;
function sortRows(data, sortField, sortDirection) {
  switch (sortField) {
    case _profilingUtils.TopNFunctionSortField.Frame:
      return (0, _lodash.orderBy)(data, row => (0, _profilingUtils.getCalleeFunction)(row.frame), sortDirection);
    case _profilingUtils.TopNFunctionSortField.SelfCPU:
      return (0, _lodash.orderBy)(data, row => row.selfCPUPerc, sortDirection);
    case _profilingUtils.TopNFunctionSortField.TotalCPU:
      return (0, _lodash.orderBy)(data, row => row.totalCPUPerc, sortDirection);
    default:
      return (0, _lodash.orderBy)(data, sortField, sortDirection);
  }
}
function getFrameIdentification(frame) {
  return [frame.SourceFilename, frame.FunctionName, frame.ExeFileName, frame.FileID, frame.AddressOrLine].join('|');
}
function DifferentialTopNFunctionsGrid({
  base,
  baselineScaleFactor,
  comparison,
  comparisonScaleFactor,
  onChangePage,
  onChangeSort,
  pageIndex,
  sortDirection,
  sortField,
  totalSeconds,
  onFrameClick,
  comparisonSortDirection,
  comparisonSortField
}) {
  const calculateImpactEstimates = (0, _use_calculate_impact_estimates.useCalculateImpactEstimate)();
  const [selectedFrame, setSelectedFrame] = (0, _react2.useState)();
  const theme = (0, _eui.useEuiTheme)();
  const totalCount = (0, _react2.useMemo)(() => {
    if (!base || !base.TotalCount) {
      return 0;
    }
    return base.TotalCount;
  }, [base]);
  function onSort(newSortingColumns) {
    // As newSortingColumns is an array and we only sort by a single field for both base and comparison
    // I need to look for the item that is not the same as in the URL to identify what's the side being sorted.
    const sortingItem = newSortingColumns.reverse().find(item => {
      const isComparison = isComparisonColumn(item.id);
      if (isComparison) {
        return !(comparisonSortField === item.id && comparisonSortDirection === item.direction);
      }
      return !(sortField === item.id && sortDirection === item.direction);
    });
    if (sortingItem) {
      const isComparison = isComparisonColumn(sortingItem.id);
      onChangeSort(isComparison ? {
        comparisonSortDirection: sortingItem.direction,
        comparisonSortField: sortingItem.id
      } : {
        sortDirection: sortingItem.direction,
        sortField: sortingItem.id
      });
    }
  }
  const {
    baseRows,
    comparisonRows
  } = (0, _react2.useMemo)(() => {
    return {
      baseRows: (0, _utils.getFunctionsRows)({
        calculateImpactEstimates,
        topNFunctions: base,
        totalSeconds
      }),
      comparisonRows: (0, _utils.getFunctionsRows)({
        baselineScaleFactor,
        calculateImpactEstimates,
        comparisonScaleFactor,
        comparisonTopNFunctions: base,
        topNFunctions: comparison,
        totalSeconds
      })
    };
  }, [base, baselineScaleFactor, calculateImpactEstimates, comparison, comparisonScaleFactor, totalSeconds]);
  const columns = (0, _react2.useMemo)(() => {
    const compareFrameAction = (0, _get_compare_frame_action.getCompareFrameAction)({
      baseRows,
      comparisonRows,
      onClick: setSelectedFrame,
      selectedFrame
    });
    return (0, _get_columns.getColumns)(compareFrameAction);
  }, [baseRows, comparisonRows, selectedFrame]);
  const sortedBaseRows = (0, _react2.useMemo)(() => {
    return sortRows(baseRows, sortField, sortDirection);
  }, [baseRows, sortDirection, sortField]);
  const sortedComparisonRows = (0, _react2.useMemo)(() => {
    return sortRows(comparisonRows, removeComparisonFromId(comparisonSortField), comparisonSortDirection);
  }, [comparisonRows, comparisonSortDirection, comparisonSortField]);
  const [visibleColumns, setVisibleColumns] = (0, _react2.useState)(columns.map(({
    id
  }) => id));
  function CellValue({
    rowIndex,
    columnId,
    setCellProps
  }) {
    const isComparison = isComparisonColumn(columnId);
    const data = isComparison ? sortedComparisonRows[rowIndex] : sortedBaseRows[rowIndex];
    (0, _react2.useEffect)(() => {
      // Add thick border to divide the baseline and comparison columns
      if (isComparison && columnId === _profilingUtils.TopNComparisonFunctionSortField.ComparisonRank) {
        setCellProps({
          style: {
            borderLeft: theme.euiTheme.border.thick
          }
        });
      } else if (columnId === _profilingUtils.TopNFunctionSortField.TotalCPU) {
        setCellProps({
          style: {
            borderRight: theme.euiTheme.border.thin
          }
        });
      }
    }, [columnId, isComparison, setCellProps]);
    if (data === undefined) {
      return null;
    }
    return /*#__PURE__*/_react2.default.createElement("div", {
      "data-test-subj": columnId
    }, /*#__PURE__*/_react2.default.createElement(_function_row.FunctionRow, {
      functionRow: data,
      columnId: removeComparisonFromId(columnId),
      setCellProps: setCellProps,
      totalCount: totalCount,
      onFrameClick: onFrameClick
    }));
  }
  const rowCount = Math.min(Math.max(sortedBaseRows.length, sortedComparisonRows.length), 100);
  return /*#__PURE__*/_react2.default.createElement(_eui.EuiDataGrid, {
    "data-test-subj": "profilingDiffTopNFunctionsGrid",
    css: (0, _react.css)`
        .thickBorderLeft {
          border-left: ${theme.euiTheme.border.thick} !important;
        }
      `,
    "aria-label": _i18n.i18n.translate('xpack.profiling.onWeelkDiffTopN.euiDataGrid.topNFunctionsLabel', {
      defaultMessage: 'TopN functions'
    }),
    columns: columns,
    columnVisibility: {
      visibleColumns,
      setVisibleColumns
    },
    rowCount: rowCount,
    renderCellValue: CellValue,
    sorting: {
      columns: [{
        id: sortField,
        direction: sortDirection
      }, {
        id: comparisonSortField,
        direction: comparisonSortDirection
      }],
      onSort
    },
    pagination: {
      pageIndex,
      pageSize: 50,
      // Left it empty on purpose as it is a required property on the pagination
      onChangeItemsPerPage: () => {},
      onChangePage,
      pageSizeOptions: []
    },
    rowHeightsOptions: {
      defaultHeight: 'auto'
    },
    toolbarVisibility: {
      showColumnSelector: false,
      showKeyboardShortcuts: false,
      showDisplaySelector: false,
      showSortSelector: false
    }
  });
}