"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.sloRouteRepository = void 0;
var _elasticsearch = require("@elastic/elasticsearch");
var _boom = require("@hapi/boom");
var _sloSchema = require("@kbn/slo-schema");
var _slo = require("../../services/slo");
var _fetch_historical_summary = require("../../services/slo/fetch_historical_summary");
var _find_slo_definitions = require("../../services/slo/find_slo_definitions");
var _get_burn_rates = require("../../services/slo/get_burn_rates");
var _get_diagnosis = require("../../services/slo/get_diagnosis");
var _get_preview_data = require("../../services/slo/get_preview_data");
var _get_slo_instances = require("../../services/slo/get_slo_instances");
var _historical_summary_client = require("../../services/slo/historical_summary_client");
var _manage_slo = require("../../services/slo/manage_slo");
var _reset_slo = require("../../services/slo/reset_slo");
var _summary_search_client = require("../../services/slo/summary_search_client");
var _summary_transform_generator = require("../../services/slo/summary_transform_generator/summary_transform_generator");
var _transform_generators = require("../../services/slo/transform_generators");
var _create_observability_server_route = require("../create_observability_server_route");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const transformGenerators = {
  'sli.apm.transactionDuration': new _transform_generators.ApmTransactionDurationTransformGenerator(),
  'sli.apm.transactionErrorRate': new _transform_generators.ApmTransactionErrorRateTransformGenerator(),
  'sli.kql.custom': new _transform_generators.KQLCustomTransformGenerator(),
  'sli.metric.custom': new _transform_generators.MetricCustomTransformGenerator(),
  'sli.histogram.custom': new _transform_generators.HistogramTransformGenerator(),
  'sli.metric.timeslice': new _transform_generators.TimesliceMetricTransformGenerator()
};
const assertPlatinumLicense = async context => {
  const licensing = await context.licensing;
  const hasCorrectLicense = licensing.license.hasAtLeast('platinum');
  if (!hasCorrectLicense) {
    throw (0, _boom.forbidden)('Platinum license or higher is needed to make use of this feature.');
  }
};
const createSLORoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'POST /api/observability/slos 2023-10-31',
  options: {
    tags: ['access:slo_write'],
    access: 'public'
  },
  params: _sloSchema.createSLOParamsSchema,
  handler: async ({
    context,
    params,
    logger,
    dependencies,
    request
  }) => {
    var _await$dependencies$s, _await$dependencies$s2, _dependencies$spaces, _dependencies$spaces$;
    await assertPlatinumLicense(context);
    const spaceId = (_await$dependencies$s = (_await$dependencies$s2 = await ((_dependencies$spaces = dependencies.spaces) === null || _dependencies$spaces === void 0 ? void 0 : (_dependencies$spaces$ = _dependencies$spaces.spacesService) === null || _dependencies$spaces$ === void 0 ? void 0 : _dependencies$spaces$.getActiveSpace(request))) === null || _await$dependencies$s2 === void 0 ? void 0 : _await$dependencies$s2.id) !== null && _await$dependencies$s !== void 0 ? _await$dependencies$s : 'default';
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const soClient = (await context.core).savedObjects.client;
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient, logger);
    const transformManager = new _slo.DefaultTransformManager(transformGenerators, esClient, logger);
    const summaryTransformManager = new _slo.DefaultSummaryTransformManager(new _summary_transform_generator.DefaultSummaryTransformGenerator(), esClient, logger);
    const createSLO = new _slo.CreateSLO(esClient, repository, transformManager, summaryTransformManager, logger, spaceId);
    const response = await createSLO.execute(params.body);
    return response;
  }
});
const updateSLORoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'PUT /api/observability/slos/{id} 2023-10-31',
  options: {
    tags: ['access:slo_write'],
    access: 'public'
  },
  params: _sloSchema.updateSLOParamsSchema,
  handler: async ({
    context,
    request,
    params,
    logger,
    dependencies
  }) => {
    var _await$dependencies$s3, _await$dependencies$s4, _dependencies$spaces2, _dependencies$spaces3;
    await assertPlatinumLicense(context);
    const spaceId = (_await$dependencies$s3 = (_await$dependencies$s4 = await ((_dependencies$spaces2 = dependencies.spaces) === null || _dependencies$spaces2 === void 0 ? void 0 : (_dependencies$spaces3 = _dependencies$spaces2.spacesService) === null || _dependencies$spaces3 === void 0 ? void 0 : _dependencies$spaces3.getActiveSpace(request))) === null || _await$dependencies$s4 === void 0 ? void 0 : _await$dependencies$s4.id) !== null && _await$dependencies$s3 !== void 0 ? _await$dependencies$s3 : 'default';
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const soClient = (await context.core).savedObjects.client;
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient, logger);
    const transformManager = new _slo.DefaultTransformManager(transformGenerators, esClient, logger);
    const summaryTransformManager = new _slo.DefaultSummaryTransformManager(new _summary_transform_generator.DefaultSummaryTransformGenerator(), esClient, logger);
    const updateSLO = new _slo.UpdateSLO(repository, transformManager, summaryTransformManager, esClient, logger, spaceId);
    const response = await updateSLO.execute(params.path.id, params.body);
    return response;
  }
});
const deleteSLORoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'DELETE /api/observability/slos/{id} 2023-10-31',
  options: {
    tags: ['access:slo_write'],
    access: 'public'
  },
  params: _sloSchema.deleteSLOParamsSchema,
  handler: async ({
    request,
    context,
    params,
    logger,
    dependencies: {
      getRulesClientWithRequest
    }
  }) => {
    await assertPlatinumLicense(context);
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const soClient = (await context.core).savedObjects.client;
    const rulesClient = getRulesClientWithRequest(request);
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient, logger);
    const transformManager = new _slo.DefaultTransformManager(transformGenerators, esClient, logger);
    const summaryTransformManager = new _slo.DefaultSummaryTransformManager(new _summary_transform_generator.DefaultSummaryTransformGenerator(), esClient, logger);
    const deleteSLO = new _slo.DeleteSLO(repository, transformManager, summaryTransformManager, esClient, rulesClient);
    await deleteSLO.execute(params.path.id);
  }
});
const getSLORoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'GET /api/observability/slos/{id} 2023-10-31',
  options: {
    tags: ['access:slo_read'],
    access: 'public'
  },
  params: _sloSchema.getSLOParamsSchema,
  handler: async ({
    context,
    params,
    logger
  }) => {
    await assertPlatinumLicense(context);
    const soClient = (await context.core).savedObjects.client;
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient, logger);
    const summaryClient = new _slo.DefaultSummaryClient(esClient);
    const getSLO = new _slo.GetSLO(repository, summaryClient);
    const response = await getSLO.execute(params.path.id, params.query);
    return response;
  }
});
const enableSLORoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'POST /api/observability/slos/{id}/enable 2023-10-31',
  options: {
    tags: ['access:slo_write'],
    access: 'public'
  },
  params: _sloSchema.manageSLOParamsSchema,
  handler: async ({
    context,
    params,
    logger
  }) => {
    await assertPlatinumLicense(context);
    const soClient = (await context.core).savedObjects.client;
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient, logger);
    const transformManager = new _slo.DefaultTransformManager(transformGenerators, esClient, logger);
    const summaryTransformManager = new _slo.DefaultSummaryTransformManager(new _summary_transform_generator.DefaultSummaryTransformGenerator(), esClient, logger);
    const manageSLO = new _manage_slo.ManageSLO(repository, transformManager, summaryTransformManager);
    const response = await manageSLO.enable(params.path.id);
    return response;
  }
});
const disableSLORoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'POST /api/observability/slos/{id}/disable 2023-10-31',
  options: {
    tags: ['access:slo_write'],
    access: 'public'
  },
  params: _sloSchema.manageSLOParamsSchema,
  handler: async ({
    context,
    params,
    logger
  }) => {
    await assertPlatinumLicense(context);
    const soClient = (await context.core).savedObjects.client;
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient, logger);
    const transformManager = new _slo.DefaultTransformManager(transformGenerators, esClient, logger);
    const summaryTransformManager = new _slo.DefaultSummaryTransformManager(new _summary_transform_generator.DefaultSummaryTransformGenerator(), esClient, logger);
    const manageSLO = new _manage_slo.ManageSLO(repository, transformManager, summaryTransformManager);
    const response = await manageSLO.disable(params.path.id);
    return response;
  }
});
const resetSLORoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'POST /api/observability/slos/{id}/_reset 2023-10-31',
  options: {
    tags: ['access:slo_write'],
    access: 'public'
  },
  params: _sloSchema.resetSLOParamsSchema,
  handler: async ({
    context,
    request,
    params,
    logger,
    dependencies
  }) => {
    var _await$dependencies$s5, _await$dependencies$s6, _dependencies$spaces4, _dependencies$spaces5;
    await assertPlatinumLicense(context);
    const spaceId = (_await$dependencies$s5 = (_await$dependencies$s6 = await ((_dependencies$spaces4 = dependencies.spaces) === null || _dependencies$spaces4 === void 0 ? void 0 : (_dependencies$spaces5 = _dependencies$spaces4.spacesService) === null || _dependencies$spaces5 === void 0 ? void 0 : _dependencies$spaces5.getActiveSpace(request))) === null || _await$dependencies$s6 === void 0 ? void 0 : _await$dependencies$s6.id) !== null && _await$dependencies$s5 !== void 0 ? _await$dependencies$s5 : 'default';
    const soClient = (await context.core).savedObjects.client;
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient, logger);
    const transformManager = new _slo.DefaultTransformManager(transformGenerators, esClient, logger);
    const summaryTransformManager = new _slo.DefaultSummaryTransformManager(new _summary_transform_generator.DefaultSummaryTransformGenerator(), esClient, logger);
    const resetSLO = new _reset_slo.ResetSLO(esClient, repository, transformManager, summaryTransformManager, logger, spaceId);
    const response = await resetSLO.execute(params.path.id);
    return response;
  }
});
const findSLORoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'GET /api/observability/slos 2023-10-31',
  options: {
    tags: ['access:slo_read'],
    access: 'public'
  },
  params: _sloSchema.findSLOParamsSchema,
  handler: async ({
    context,
    request,
    params,
    logger,
    dependencies
  }) => {
    var _await$dependencies$s7, _await$dependencies$s8, _dependencies$spaces6, _dependencies$spaces7, _params$query;
    await assertPlatinumLicense(context);
    const spaceId = (_await$dependencies$s7 = (_await$dependencies$s8 = await ((_dependencies$spaces6 = dependencies.spaces) === null || _dependencies$spaces6 === void 0 ? void 0 : (_dependencies$spaces7 = _dependencies$spaces6.spacesService) === null || _dependencies$spaces7 === void 0 ? void 0 : _dependencies$spaces7.getActiveSpace(request))) === null || _await$dependencies$s8 === void 0 ? void 0 : _await$dependencies$s8.id) !== null && _await$dependencies$s7 !== void 0 ? _await$dependencies$s7 : 'default';
    const soClient = (await context.core).savedObjects.client;
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient, logger);
    const summarySearchClient = new _summary_search_client.DefaultSummarySearchClient(esClient, logger, spaceId);
    const findSLO = new _slo.FindSLO(repository, summarySearchClient);
    const response = await findSLO.execute((_params$query = params === null || params === void 0 ? void 0 : params.query) !== null && _params$query !== void 0 ? _params$query : {});
    return response;
  }
});
const deleteSloInstancesRoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'POST /api/observability/slos/_delete_instances 2023-10-31',
  options: {
    tags: ['access:slo_write']
  },
  params: _sloSchema.deleteSLOInstancesParamsSchema,
  handler: async ({
    context,
    params
  }) => {
    await assertPlatinumLicense(context);
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const deleteSloInstances = new _slo.DeleteSLOInstances(esClient);
    await deleteSloInstances.execute(params.body);
  }
});
const findSloDefinitionsRoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'GET /api/observability/slos/_definitions 2023-10-31',
  options: {
    tags: ['access:slo_read']
  },
  params: _sloSchema.findSloDefinitionsParamsSchema,
  handler: async ({
    context,
    params,
    logger
  }) => {
    var _params$query2;
    await assertPlatinumLicense(context);
    const soClient = (await context.core).savedObjects.client;
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient, logger);
    const findSloDefinitions = new _find_slo_definitions.FindSLODefinitions(repository);
    const response = await findSloDefinitions.execute((_params$query2 = params === null || params === void 0 ? void 0 : params.query) !== null && _params$query2 !== void 0 ? _params$query2 : {});
    return response;
  }
});
const fetchHistoricalSummary = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'POST /internal/observability/slos/_historical_summary',
  options: {
    tags: ['access:slo_read']
  },
  params: _sloSchema.fetchHistoricalSummaryParamsSchema,
  handler: async ({
    context,
    params,
    logger
  }) => {
    await assertPlatinumLicense(context);
    const soClient = (await context.core).savedObjects.client;
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient, logger);
    const historicalSummaryClient = new _historical_summary_client.DefaultHistoricalSummaryClient(esClient);
    const fetchSummaryData = new _fetch_historical_summary.FetchHistoricalSummary(repository, historicalSummaryClient);
    const response = await fetchSummaryData.execute(params.body);
    return response;
  }
});
const getSLOInstancesRoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'GET /internal/observability/slos/{id}/_instances',
  options: {
    tags: ['access:slo_read'],
    access: 'internal'
  },
  params: _sloSchema.getSLOInstancesParamsSchema,
  handler: async ({
    context,
    params,
    logger
  }) => {
    await assertPlatinumLicense(context);
    const soClient = (await context.core).savedObjects.client;
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const repository = new _slo.KibanaSavedObjectsSLORepository(soClient, logger);
    const getSLOInstances = new _get_slo_instances.GetSLOInstances(repository, esClient);
    const response = await getSLOInstances.execute(params.path.id);
    return response;
  }
});
const getDiagnosisRoute = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'GET /internal/observability/slos/_diagnosis',
  options: {
    tags: [],
    access: 'internal'
  },
  params: undefined,
  handler: async ({
    context
  }) => {
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const licensing = await context.licensing;
    try {
      const response = await (0, _get_diagnosis.getGlobalDiagnosis)(esClient, licensing);
      return response;
    } catch (error) {
      if (error instanceof _elasticsearch.errors.ResponseError && error.statusCode === 403) {
        throw (0, _boom.forbidden)('Insufficient Elasticsearch cluster permissions to access feature.');
      }
      throw (0, _boom.failedDependency)(error);
    }
  }
});
const getSloBurnRates = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'POST /internal/observability/slos/{id}/_burn_rates',
  options: {
    tags: ['access:slo_read'],
    access: 'internal'
  },
  params: _sloSchema.getSLOBurnRatesParamsSchema,
  handler: async ({
    context,
    params,
    logger
  }) => {
    await assertPlatinumLicense(context);
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const soClient = (await context.core).savedObjects.client;
    const burnRates = await (0, _get_burn_rates.getBurnRates)(params.path.id, params.body.instanceId, params.body.windows, {
      soClient,
      esClient,
      logger
    });
    return {
      burnRates
    };
  }
});
const getPreviewData = (0, _create_observability_server_route.createObservabilityServerRoute)({
  endpoint: 'POST /internal/observability/slos/_preview',
  options: {
    tags: ['access:slo_read'],
    access: 'internal'
  },
  params: _sloSchema.getPreviewDataParamsSchema,
  handler: async ({
    context,
    params
  }) => {
    await assertPlatinumLicense(context);
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const service = new _get_preview_data.GetPreviewData(esClient);
    return await service.execute(params.body);
  }
});
const sloRouteRepository = {
  ...createSLORoute,
  ...deleteSLORoute,
  ...deleteSloInstancesRoute,
  ...disableSLORoute,
  ...enableSLORoute,
  ...fetchHistoricalSummary,
  ...findSloDefinitionsRoute,
  ...findSLORoute,
  ...getSLORoute,
  ...updateSLORoute,
  ...getDiagnosisRoute,
  ...getSloBurnRates,
  ...getPreviewData,
  ...getSLOInstancesRoute,
  ...resetSLORoute
};
exports.sloRouteRepository = sloRouteRepository;